<?php

namespace Inside\Archive\Services;

use Illuminate\Support\Facades\Event;
use Inside\Archive\Events\ContentArchivedEvent;
use Inside\Archive\Events\ContentRestoredEvent;
use Inside\Archive\Models\Archive;
use Inside\Content\Facades\ContentCleaner;
use Inside\Content\Facades\Schema;
use Inside\Content\Jobs\DeleteContent;
use Inside\Content\Listeners\BaseContentListener;
use Inside\Content\Models\Content;
use Inside\Host\Bridge\BridgeContent;

class ArchiveService extends BaseContentListener
{
    public function isArchived(string $type, string $uuid): bool
    {
        return Archive::where('type', $type)
            ->where('uuid', $uuid)
            ->where('date', '<=', date('Y-m-d H:i:s'))
            ->exists();
    }

    public function archiveContent(
        string $type,
        string $uuid,
        int $date = null,
        bool $unPublish = false,
        bool $stopPropagation = false
    ): void {
        if (is_null($date)) {
            $date = strtotime(date('Y-m-d'));
        }

        $date = (int) $date;
        $content = call_user_func($type.'::find', $uuid);
        $options = Schema::getModelOptions(class_to_type($content));
        $deletionStrategy = $options['deletion_strategy'] ?? null;

        if (
            ContentCleaner::shouldDeleteChildren($content)
            && isset($options['deletion_strategy'])
            && in_array($deletionStrategy, ['parent', 'only_parent'])
        ) {
            ContentCleaner::getChildren($content, $deletionStrategy == 'only_parent')
                ->each(function (Content $child) use ($date, $unPublish, $stopPropagation) {
                    ContentCleaner::forceChildrenDeletion($child, true);
                    $this->archiveContent(
                        type_to_class($child->content_type),
                        (string) $child->uuid,
                        $date,
                        $unPublish,
                        $stopPropagation
                    );
                });
        }

        $this->createOrUpdateArchive($type, $uuid, $date, $unPublish, $stopPropagation);
    }

    public function restoreContent(
        string $type,
        string $uuid,
        bool $publish = false,
        bool $stopPropagation = false
    ): void {
        Archive::where('type', $type)
            ->where('uuid', $uuid)
            ->delete();

        if ($publish) {
            $content = call_user_func($type . '::find', $uuid);

            if ($content) {
                $bridge = new BridgeContent();

                $bridge->setStatus($content, true);
                $this->forgetCache($content);

                if (!$stopPropagation) {
                    Event::dispatch(new ContentRestoredEvent($content));
                }
            }
        }
    }

    /**
     * @param string $type
     * @param string $uuid
     * @param int $date
     * @param bool $unPublish
     * @param bool $stopPropagation
     * @return void
     * @throws \Exception
     */
    public function createOrUpdateArchive(string $type, string $uuid, int $date, bool $unPublish, bool $stopPropagation): void
    {
        if ($this->isArchived($type, $uuid)) {
            return;
        }
        Archive::updateOrCreate(
            [
                'type' => $type,
                'uuid' => $uuid,
            ],
            [
                'date' => date('Y-m-d H:i:s', $date),
            ]
        );

        if ($unPublish && date('Y-m-d') === date('Y-m-d', $date)) {
            $content = call_user_func($type . '::find', $uuid);

            if (!is_null($content)) {
                $bridge = new BridgeContent();

                $bridge->setStatus($content, false);
                $this->forgetCache($content);

                if (!$stopPropagation) {
                    Event::dispatch(new ContentArchivedEvent($content));
                }
            }
        }
    }
}
