<?php

declare(strict_types=1);

namespace Inside\Archive\Http\Controllers;

use Illuminate\Http\JsonResponse;
use Illuminate\Http\Request;
use Illuminate\Support\Arr;
use Illuminate\Support\Facades\DB;
use Illuminate\Support\Facades\Event;
use Illuminate\Support\Facades\Queue;
use Inside\Archive\Events\ContentRestoredEvent;
use Inside\Archive\Models\Archive;
use Inside\Archive\Services\ArchiveService;
use Inside\Content\Facades\ContentHelper;
use Inside\Content\Services\Queries\ContentQuery;
use Inside\Host\Bridge\BridgeContent;

final class ArchiveController
{
    public function __construct(
        private ArchiveService $archiveService
    ) {
    }

    public function list(Request $request): JsonResponse
    {
        $service = new ContentQuery();
        $filters = ContentHelper::extractFiltersInputFromRequest($request);
        $fields = ContentHelper::extractFieldsInputFromRequest($request);

        if ($request->has('query')) {
            $filters = array_merge(
                $filters,
                [
                    'title:like' => '%'.str_replace(' ', '%', $request->get('query')).'%',
                ]
            );
        }

        $fields = array_merge($fields, [
            'uuid', 'archive', 'content_type', 'title', 'status', 'created_at', 'updated_at', 'slug',
            ['authors' => ['firstname', 'lastname', 'image', 'uuid']],
        ]);

        $request->merge([
            'fields' => json_encode($fields),
            'filters' => json_encode($filters),
        ]);

        $service->requestQuery('_contents', $request);
        $data = $service->transformCollection();

        foreach ($data['data'] as &$content) {
            $date = DB::table('inside_archives')
                ->where('uuid', $content['uuid'])
                ->value('date');

            $content['archived_at'] = $date !== null ? strtotime($date) : null;
        }

        return response()->json($data);
    }

    public function restore(Request $request): JsonResponse
    {
        $query = Archive::query()->where('date', '<=', date('Y-m-d H:i:s'));

        if ($request->has('uuids')) {
            $query->whereIn('uuid', $request->get('uuids'));
        }

        $contents = $query->get()->toArray();

        foreach ($contents as $content) {
            $this->archiveService->restoreContent(
                type: $content['type'],
                uuid: $content['uuid'],
                publish: true,
                stopPropagation: false
            );
        }

        return response()->json([
            'success' => true,
        ]);
    }

    public function delete(Request $request): JsonResponse
    {
        $query = Archive::query()->where('date', '<=', date('Y-m-d H:i:s'));

        if ($request->has('uuids')) {
            $query->whereIn('uuid', $request->get('uuids'));
        }

        $contents = $query->get()->toArray();

        foreach ($contents as $content) {
            $content = call_user_func($content['type'].'::find', $content['uuid']);
            Archive::where('type', $content['type'])->where('uuid', $content['uuid'])->delete();

            if (is_null($content)) {
                continue;
            }

            $bridge = new BridgeContent();
            try {
                $bridge->contentDelete($content->content_type, $content->uuid);
            } catch (\Exception $e) {
                throw $e;
            }
        }

        return response()->json([
            'success' => true,
        ]);
    }
}
