<?php

namespace Inside\Archive\Http\Middlewares;

use Closure;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Lang;
use Illuminate\Support\Str;
use Inside\Archive\Models\Archive;
use Inside\Archive\Services\ArchiveService;
use Inside\Content\Facades\ContentCleaner;
use Inside\Content\Services\QuickAccessService;
use Inside\Permission\Exceptions\AuthorizationException;
use Inside\Permission\Facades\Permission;
use Inside\Slug\Services\SlugService;
use Symfony\Component\HttpFoundation\Response;

/**
 * Archive form middleware
 *
 * @category Class
 * @package  Inside\Archive\Http\Middleware\ArchiveFormMiddleware
 * @author   Maecia <technique@maecia.com>
 * @license  http://www.gnu.org/copyleft/gpl.html GNU General Public License
 * @link     http://www.maecia.com/
 */
class ArchiveFormMiddleware
{
    /**
     * Handle an incoming request.
     */
    public function handle(Request $request, Closure $next): mixed
    {
        $path = $request->path();
        $archives = null;
        $service = new ArchiveService();
        $config = config('archive');

        if (str_starts_with($path, 'api/v1/content/asset')) {
            return $next($request);
        }

        if (str_starts_with($path, 'api/v1/content/') && in_array($request->getMethod(), ['POST', 'PUT']) && $request->has('archives')) {
            $archives = $request->get('archives');
            unset($request['archives']);

            $response = $next($request);
            if ($response instanceof Response && $response->getStatusCode() != 200) {
                return $response;
            }

            $route = (array) $request->route();
            if (! isset($route[2]['type'])) {
                return $response;
            }

            if (! empty($config['excludeContentTypes']) && in_array($route[2]['type'], $config['excludeContentTypes'])) {
                return $response;
            }

            if (! empty($config['contentTypes']) && ! in_array($route[2]['type'], $config['contentTypes'])) {
                return $response;
            }

            /** @var array $content */
            $content = json_decode_response($response);
            $class = type_to_class($route[2]['type']);

            if (! $archives['enabled']) {
                $service->restoreContent($class, $content['uuid']);
            } else {
                $service->archiveContent($class, $content['uuid'], $archives['date']);
            }

            return $response;
        }

        // Archive content instead of deleting it
        if (str_starts_with($path, 'api/v1/content') && $request->method() === 'DELETE') {
            $contentType = str_before(str_after($path, 'content/'), '/');
            if (! $config['overrideDelete'] || in_array($contentType, $config['excludeContentTypes'])) {
                return $next($request);
            }

            $contentType = '';
            $deleteFails = [];

            if (str_starts_with($path, 'api/v1/content/delete')) {
                $data = $request->get('contents');
                $resultUuids = [
                    'succeed' => [],
                    'failed' => [],
                ];
                foreach ($data as $contentType => $uuids) {
                    foreach ($uuids as $uuid) {
                        if (! Permission::allowed('delete', $contentType, $uuid)) {
                            $deleteFails[] = $contentType;
                        } else {
                            $this->archiveDeletingContent($request, $next, $service, $contentType, $uuid);
                            array_push($resultUuids['succeed'], $uuid);
                        }
                    }
                }
                $uuid = $resultUuids;
            } else {
                [$contentType, $uuid] = explode('/', str_replace('api/v1/content/', '', $path));
                $this->archiveDeletingContent($request, $next, $service, $contentType, $uuid);
            }

            $this->cleanQuickAccess($contentType, $uuid);

            if (! empty($deleteFails)) {
                return response()->json([
                    'error' => 'Vous n\'avez pas la permission de supprimer ces contenus',
                    'type' => $deleteFails,
                ], 403);
            }

            $slugService = new SlugService();

            return response()->json([
                'parentUrl' => is_string($uuid) ? $slugService->getParentSlug($contentType, $uuid) : null,
                'uuids' => $uuid,
            ]);
        }

        $response = $next($request);
        if ($response instanceof Response && $response->getStatusCode() != 200) {
            return $response;
        }

        if (! Str::contains($path, 'api/v1/form/')) {
            return $response;
        }

        /** @var array|null $data */
        $data = json_decode_response($response);

        if ($data === null) {
            return $response;
        }

        if (! $config['visible']) {
            return $response;
        }

        $route = (array) $request->route();

        if (isset($route[2]['type'])) {
            if (! empty($config['excludeContentTypes']) && in_array($route[2]['type'], $config['excludeContentTypes'])) {
                return $response;
            }

            if (! empty($config['contentTypes']) && ! in_array($route[2]['type'], $config['contentTypes'])) {
                return $response;
            }
        }

        $enabled = $config['enabled'];
        $date = strtotime($config['date'], (int) strtotime(date('Y-m-d')));

        if (isset($route[2]['id'])) {
            $content = call_user_func(type_to_class($route[2]['type']).'::find', $route[2]['id']);

            if (! $content) {
                return $response;
            }

            $archive = Archive::where('type', type_to_class($route[2]['type']))->where('uuid', $route[2]['id'])->first();

            if (! $archive) {
                $enabled = false;
            } else {
                $enabled = true;
                $date = strtotime($archive->date);
            }
        }

        $result = [
            'id' => 'group_archives',
            'weight' => 200,
            'type' => 'fieldset',
            'classes' => '',
            'label' => [Lang::getLocale() => Lang::get('fieldset.archiving')],
            'fields' => [
                [
                    'name' => 'archives',
                    'type' => 'archives',
                    'enabled' => $enabled,
                    'date' => $date,
                    // TODO: remove options when front is done
                    'options' => [
                        'enabled' => $enabled,
                        'date' => $date,
                    ],
                ],
            ],
        ];

        $data['data'][] = $result;
        set_response($response, $data);

        return $response;
    }

    protected function cleanQuickAccess(string $contentType, string|array $uuid): void
    {
        if (
            'tools' !== $contentType ||
            !is_string($uuid)
        ) {
            return;
        }
        $quickAccessService = new QuickAccessService();
        $quickAccessService->cleanUnpublishedTools();
    }
    /**
     * @param Request $request
     * @param Closure $next
     * @param ArchiveService $service
     * @param string $contentType
     * @param string $uuid
     *
     * @return false|mixed
     * @throws \Exception
     */
    protected function archiveDeletingContent(Request $request, Closure $next, ArchiveService $service, string $contentType, string $uuid)
    {
        $config = config('archive');
        $route = (array) $request->route();

        if (! empty($config['excludeContentTypes']) && in_array($contentType, $config['excludeContentTypes'])) {
            return $next($request);
        }

        if (! empty($config['contentTypes']) && ! in_array($contentType, $config['contentTypes'])) {
            return $next($request);
        }

        if ($service->isArchived(type_to_class($contentType), $uuid)) {
            return $next($request);
        }

        $content = call_user_func(type_to_class($contentType).'::find', $uuid);

        if ($content) {
            ContentCleaner::forceChildrenDeletion(
                $content,
                $request->get('force_children_deletion') ?? $request->query->getBoolean('force_children_deletion')
            );
            $service->archiveContent(type_to_class($contentType), $uuid, null, true);
        }

        return $content;
    }
}
