<?php

namespace Inside\Form\Advanced\Services;

use Illuminate\Support\Facades\Storage;
use Inside\Form\Advanced\Events\AdvancedFormSubmissionExportedEvent;
use Inside\Form\Advanced\Exceptions\EmptySubmissionListException;
use Inside\Form\Advanced\Exports\AdvancedFormExport;
use Inside\Host\Bridge\BridgeContent;
use Inside\Support\Str;
use Maatwebsite\Excel\Facades\Excel;
use ZipArchive;

class AdvancedFormsExportService
{
    private array $submissionsUuids = [];
    private ?string $userUuid = null;

    public function setSubmissionsUuids(array $uuids): self
    {
        $this->submissionsUuids = $uuids;
        return $this;
    }

    public function setUserUuid(?string $uuid): self
    {
        $this->userUuid = $uuid;
        return $this;
    }

    public function execute(): void
    {
        $submissions = type_to_class('advanced_form_submissions')::query()
            ->whereIn('uuid', $this->submissionsUuids)
            ->get();

        if ($submissions->isEmpty()) {
            throw new EmptySubmissionListException('Aucune soumission trouvée.');
        }

        $byForm = $submissions->groupBy(function ($s) {
            $out = is_string($s->output) ? json_decode($s->output, true) : (array)$s->output;
            return $out['advanced_forms'] ?? 'unknown';
        });


        $disk = Storage::disk('local');
        $workdir = 'exports/advanced-forms';
        if (!$disk->exists($workdir)) {
            $disk->makeDirectory($workdir);
        }

        $paths = [];

        foreach ($byForm as $formUuid => $group) {
            $form = type_to_class('advanced_forms')::find($formUuid);

            $base = $form?->title ?? ('form_'.$formUuid);
            $safe = Str::slug($base, '_').'_'.now()->format('Ymd_His');
            $filename = "{$safe}.xlsx";
            $path = "{$workdir}/{$filename}";

            Excel::store(
                new AdvancedFormExport($form, $group),
                $path,
                'local'
            );

            $paths[] = $path;
        }

        $exportableFilePath = $paths[0];

        if (count($paths) > 1) {
            $zipName = 'advanced_forms_exports_'.now()->format('Ymd_His').'.zip';
            $exportableFilePath = $workdir.'/'.$zipName;

            $zip = new ZipArchive();
            $zip->open($disk->path($exportableFilePath), ZipArchive::CREATE | ZipArchive::OVERWRITE);

            foreach ($paths as $path) {
                $zip->addFile($disk->path($path), basename($path));
            }
            $zip->close();
            foreach ($paths as $path) {
                $disk->delete($path);
            }
        }

        $this->notify($exportableFilePath);
    }

    private function notify(string $exportableFilePath): void
    {
        if (!empty($this->userUuid)) {
            event(new AdvancedFormSubmissionExportedEvent($this->userUuid, $exportableFilePath));
        }
    }
}
