<?php

declare(strict_types=1);

namespace Inside\Form\Advanced\Listeners;

use Illuminate\Support\Facades\Log;
use Illuminate\Contracts\Queue\ShouldQueue;
use Inside\Authentication\Models\User;
use Inside\Content\Events\ContentCreatedEvent;
use Inside\Content\Models\Contents\AdvancedFormSubmissions;
use Inside\Notify\Listeners\BaseNotificationListener;
use Inside\Notify\Models\NotificationType;
use Inside\Permission\Models\Role;

final class AdvancedFormSubmittedListener extends BaseNotificationListener implements ShouldQueue
{
    public function handle(ContentCreatedEvent $event): void
    {
        $advancedFormSubmission = $event->model;
        if ($advancedFormSubmission->content_type != 'advanced_form_submissions') {
            return;
        }

        $advancedForm = $advancedFormSubmission->advancedForms()->get()->first();
        $advancedFormSubmissionAuthor = User::find($advancedFormSubmission->author_id);
        if (empty($advancedForm->admin_roles)) {
            $roles = [Role::where(['name' => 'super_administrator'])->first()?->id];
        } else {
            $roles = explode(',', $advancedForm->admin_roles);
        }

        /**
         * @var \Illuminate\Database\Eloquent\Collection<int, User> $users
         */
        $users = User::whereHas('roles', fn ($query) => $query->whereIn('role_id', $roles))->get();


        if (
            empty($advancedForm) ||
            $users->isEmpty() ||
            empty($advancedFormSubmissionAuthor) ||
            $advancedForm->status != true ||
            $advancedFormSubmission->status != true
        ) {
            return;
        }

        /** @var NotificationType[] $types */
        $types = NotificationType::where('action', 'advanced-form-submitted')
            ->get();

        foreach ($types as $type) {
            $users->each(function ($user) use ($type, $advancedForm, $roles) {
                $this->when = null;
                $this->route = $user;
                $this->notify(
                    type: $type,
                    user: $user,
                    model: $advancedForm,
                    data: [
                        'form_title' => $advancedForm->title,
                    ],
                    roles: $roles,
                    theme: null
                );
            });
        }
    }
}
