<?php

namespace Inside\Form\Advanced\Listeners;

use Illuminate\Contracts\Queue\ShouldQueue;
use Illuminate\Support\Facades\Lang;
use Illuminate\Support\Facades\Log;
use Inside\Authentication\Models\User;
use Inside\Content\Events\ContentUpdatedEvent;
use Inside\Notify\Listeners\BaseNotificationListener;
use Inside\Notify\Models\NotificationType;

class AdvancedFormSubmissionStatusEditedListener extends BaseNotificationListener implements ShouldQueue
{
    public function handle(ContentUpdatedEvent $event): void
    {
        $advancedFormSubmission = $event->model;
        if ($advancedFormSubmission->content_type != 'advanced_form_submissions') {
            return;
        }
        $advancedForm = $advancedFormSubmission->advancedForms()->get()->first();
        /**
         * @var User|null $advancedFormSubmissionEditor
         */
        $advancedFormSubmissionEditor = $event->user;
        $advancedFormSubmissionAuthorUuid = json_decode($advancedFormSubmission->output, true)['user_uuid'] ?? null;

        /**
         * @var User|null $advancedFormSubmissionAuthor
         */
        $advancedFormSubmissionAuthor = $advancedFormSubmissionAuthorUuid ? User::find($advancedFormSubmissionAuthorUuid) : null;

        if (
            empty($advancedForm) ||
            empty($advancedFormSubmissionEditor) ||
            empty($advancedFormSubmissionAuthor) ||
            $advancedForm->status != true ||
            $advancedFormSubmission->status != true ||
            $advancedFormSubmission->created_at == $advancedFormSubmission->updated_at
        ) {
            Log::warning(sprintf('[AdvancedFormSubmissionStatusEditedListener] Created at %s equals updated at %s or missing data for submission %s', $advancedFormSubmission->created_at, $advancedFormSubmission->updated_at, $advancedFormSubmissionAuthorUuid));
            return;
        }
        $via = $advancedFormSubmission->notify ? ['web', 'email'] : ['web'];

        /** @var NotificationType[] $types */
        $types = NotificationType::where('action', 'advanced-form-submission-edited')
            ->where('condition', sprintf('submission_status:%s', $advancedFormSubmission->submission_status))
            ->whereIn('via', $via)
            ->get();


        foreach ($types as $type) {
            $this->when = null;
            $this->route = $advancedFormSubmissionAuthor;
            $this->notify(
                type: $type,
                user: $advancedFormSubmissionAuthor,
                model: $advancedForm,
                data: [
                    'form_title' => $advancedForm->title,
                    'feed_back' => empty($advancedFormSubmission->feed_back) ? '' : sprintf(
                        '%s : %s',
                        Lang::get('notifications.advanced.form.comment'),
                        $advancedFormSubmission->feed_back
                    ),
                ],
                roles: []
            );
        }
    }
}
