<?php

namespace Inside\Form\Advanced\Http\Middlewares;

use Closure;
use Illuminate\Http\Request;
use Symfony\Component\HttpFoundation\Response;

class PermissionsFilteredByRoleMiddleware
{
    private const SEARCH_PATH = 'api/v1/search';
    private const ADVANCED_FORM_SUBMISSIONS_PATH = 'api/v1/content/advanced_form_submissions';
    private const PATHS = [
        self::SEARCH_PATH,
        self::ADVANCED_FORM_SUBMISSIONS_PATH,
    ];

    /**
     * @param Request $request
     * @param Closure $next
     * @return mixed
     */
    public function handle(Request $request, Closure $next): mixed
    {
        if (
            !str($request->path())->contains(self::PATHS)
        ) {
            return $next($request);
        }

        if ($request->getMethod() !== 'GET') {
            return $next($request);
        }

        $user = $request->user();

        $response = $next($request);

        $response = $this->removeUsersWithoutSubmission($response, $request);


        if ($this->isFilteredByAuthor($request)) {
            $content = $this->anonimyzer($request, $response);
            set_response($response, $content);
        }

        if (is_null($user) || $user->hasRole('super_administrator')) {
            return $response;
        }

        $roles = $user->roles->pluck('id')->toArray();

        if (is_null($roles)) {
            return $response;
        }

        $endpoint = str($request->path());
        $permissionFiler = match (true) {
            $endpoint->contains(self::ADVANCED_FORM_SUBMISSIONS_PATH) => new AdvancedFormSubmissions(),
            $endpoint->contains(self::SEARCH_PATH) => new Search(),
            default => $next($request),
        };

        $content = $permissionFiler->content($request, $response, $roles);

        set_response($response, $content);
        return $response;
    }

    private function removeUsersWithoutSubmission(mixed $response, Request $request): mixed
    {
        $filters = $request->query->get('filters');
        if (!is_string($filters) || empty($filters)) {
            return $response;
        }
        $arrayFilters = json_decode($filters, true);
        $isAdvancedForm = data_get($arrayFilters, 'advanced_forms', false);

        if (!$isAdvancedForm || $request->path() !== self::SEARCH_PATH) {
            return $response;
        }

        $content = json_decode_response($response);
        if (collect($content['data'])->isEmpty()) {
            return $response;
        }

        $usersWithSubmission = type_to_class('advanced_form_submissions')::query()
            ->distinct('author_id')
            ->pluck('author_id')
            ->toArray();

        $allowedUuids = array_map('strval', $usersWithSubmission);

        $filteredUsers = collect($content['data'])
            ->filter(function (array $user) use ($allowedUuids) {
                $userUuid     = data_get($user, 'uuid');
                return is_string($userUuid) && in_array($userUuid, $allowedUuids, true);
            })
            ->values();

        $content['data'] = $filteredUsers;
        set_response($response, $content);
        return $response;
    }

    private function isFilteredByAuthor(Request $request): bool
    {
        $filters = $request->query->get('filters');
        if (!is_string($filters) || empty($filters)) {
            return false;
        }
        $arrayFilters = json_decode($filters, true);
        $author = data_get($arrayFilters, 'authors', []);
        return !empty($author) &&
            !empty($author['uuid:eq']) &&
            ($request->user()?->uuid !== $author['uuid:eq']);
    }

    private function anonimyzer(Request $request, Response $response): array
    {
        $content = json_decode_response($response);

        if (collect($content['data'])->isEmpty()) {
            return $content;
        }

        if ($content['data'][0]['content_type'] !== 'advanced_form_submissions') {
            return $content;
        }

        $filteredData = [];
        foreach ($content['data'] as $index => $data) {
            if (data_get($data, 'advanced_forms.confidentiality', false)) {
                continue;
            }
            $filteredData[] = $data;
        }
        $content['data'] = $filteredData;
        return $content;
    }
}
