<?php

namespace Inside\Form\Advanced\Exports;

use Illuminate\Support\Collection;
use Illuminate\Support\Facades\Lang;
use Illuminate\Support\Str;
use Inside\Content\Facades\Schema;
use Inside\Content\Models\Content;
use Inside\User\Models\User;
use Maatwebsite\Excel\Concerns\FromCollection;
use Maatwebsite\Excel\Concerns\WithHeadings;
use Maatwebsite\Excel\Concerns\ShouldAutoSize;
use Maatwebsite\Excel\Concerns\WithMapping;

class AdvancedFormExport implements FromCollection, WithHeadings, ShouldAutoSize, WithMapping
{
    public function __construct(
        protected ?Content $form,
        protected Collection $submissions
    ) {
    }

    protected array $statusMap = [
        'pending'  => 'En cours',
        'approved' => 'Approuvée',
        'rejected' => 'Refusée'
    ];

    protected array $dynamicSchema = [];
    protected array $schemaByKey = [];
    protected bool $includeStatus = false;
    protected bool $isAnonymous = false;
    protected array $userInfoKeys = [];

    public function headings(): array
    {
        $this->includeStatus = (bool)($this->form?->has_status ?? false);
        $this->isAnonymous = (bool)($this->form?->confidentiality ?? false);

        $langcode = $this->form?->langcode ?? config('app.locale', 'fr');

        $fixed = [
            Lang::get('export.advanced.form.id_submission'),
            Lang::get('export.advanced.form.submitter')
        ];

        if (!$this->isAnonymous && !empty($this->form?->user_informations)) {
            $this->userInfoKeys  = $this->parseUserInfoKeys($this->form->user_informations);
            $userSchemaFields = data_get(Schema::getSchemaInformation('users'), 'fields', []);
            $fixed = array_merge($fixed, array_map(function ($fieldName) use ($userSchemaFields, $langcode) {
                return $field = data_get($userSchemaFields, $fieldName.'.options.title.'.$langcode, $fieldName);
            }, $this->userInfoKeys));
        }

        $fixed[] = Lang::get('export.advanced.form.submission_day');

        if ($this->includeStatus) {
            $fixed[] = Lang::get('export.advanced.form.status');
        }

        $this->buildDynamicSchemaFromInputs();
        $dynamicLabels = array_map(fn ($schema) => $schema['label'], $this->dynamicSchema);

        return array_merge($fixed, $dynamicLabels);
    }

    public function collection()
    {
        $ordered = $this->submissions->sortBy('created_at')->values();

        return $ordered->values()->map(function ($submission, $index) {
            $submission->__row_index = $index + 1;
            return $submission;
        });
    }

    public function map($submission): array
    {
        $row = [
            $submission->__row_index,
            $this->resolveRequester($submission),
        ];

        // Infos demandeur si non anonyme et demandé
        $row = array_merge($row, $this->mapUserInfos($submission));

        // Date de la demande
        $row[] = get_date_in_user_timezone($submission->created_at)?->format('d/m/Y H:i') ?? "";

        // Status de la demande si validation requise
        if ($this->includeStatus) {
            $row[] = $this->mapStatus($submission->submission_status ?? $submission->status ?? null);
        }

        // Questions
        $payload = $this->decodeOutput($submission->output);
        foreach ($this->dynamicSchema as $col) {
            $key  = $col['key'];
            $meta = $this->schemaByKey[$key] ?? ['label' => $col['label']];
            $raw  = $payload[$key] ?? "";
            $row[] = $this->renderAnswer($raw, $meta);
        }

        return $row;
    }

    protected function parseUserInfoKeys(string $userInformations): array
    {
        $parts = array_map('trim', explode(',', $userInformations));
        $keys  = array_values(array_filter($parts, fn ($part) => $part !== ''));
        return array_unique($keys);
    }

    protected function buildDynamicSchemaFromInputs(): void
    {
        $this->dynamicSchema = [];
        $this->schemaByKey   = [];

        $inputs = is_string($this->form?->inputs) ? json_decode($this->form->inputs, true) : [];

        foreach ($inputs as $item) {
            if (($item['type'] ?? null) !== 'question') {
                continue;
            }

            $qid     = (string)($item['id'] ?? '');
            $title   = (string)($item['title'] ?? $qid);
            $qtype   = (string)($item['question_type'] ?? '');
            $choices = $this->extractChoices($item);

            if ($qid !== '') {
                $this->pushSchemaItem($qid, $title, $qtype, $choices);
            }

            $qOpts = $item['questionOptions']['choices'] ?? [];
            foreach ($qOpts as $choice) {
                $sub = $choice['questionOptions'] ?? null;
                if (!$sub) {
                    continue;
                }

                $sqId     = (string)($sub['id'] ?? '');
                $sqTitle  = (string)($sub['title'] ?? $sqId);
                $sqType   = (string)($sub['question_type'] ?? '');
                $sqChoices = [];

                if (isset($sub['subQuestionOptions']['choices']) && is_array($sub['subQuestionOptions']['choices'])) {
                    foreach ($sub['subQuestionOptions']['choices'] as $opt) {
                        if (!isset($opt['id'])) {
                            continue;
                        }
                        $sqChoices[(string)$opt['id']] = (string)($opt['title'] ?? $opt['id']);
                    }
                }

                if ($sqId !== '') {
                    $this->pushSchemaItem($sqId, $sqTitle, $sqType, $sqChoices);
                }
            }
        }
    }

    protected function pushSchemaItem(string $key, string $label, ?string $type, array $choices): void
    {
        $this->dynamicSchema[]   = ['key' => $key, 'label' => $label, 'type' => $type, 'choices' => $choices];
        $this->schemaByKey[$key] = ['label' => $label, 'type' => $type, 'choices' => $choices];
    }

    protected function extractChoices(array $question): array
    {
        $out     = [];
        $choices = $question['questionOptions']['choices'] ?? [];

        if (is_array($choices)) {
            foreach ($choices as $choice) {
                if (!isset($choice['id'])) {
                    continue;
                }
                $out[(string)$choice['id']] = (string)($choice['title'] ?? $choice['id']);
            }
        }

        return $out;
    }

    protected function decodeOutput(string $output): array
    {
        if (is_string($output)) {
            $decodedOutput = json_decode($output, true);
            if (is_array($decodedOutput)) {
                return $decodedOutput;
            }
        }

        return [];
    }

    protected function renderAnswer(string|array $raw, array $meta): string
    {
        $type = strtolower((string)($meta['type'] ?? ''));

        return match ($type) {
            'radio', 'selectbox' => $this->renderSingleChoice($raw, $meta),
            'checkbox' => $this->renderMultiChoice($raw, $meta),
            'date' => $this->renderDate($raw),
            default => $this->renderDefault($raw, $meta),
        };
    }

    protected function renderSingleChoice(string|array $raw, array $meta): string
    {
        if (empty($raw)) {
            return '';
        }

        $choices = $meta['choices'] ?? [];
        $id = is_array($raw) ? ($raw['id'] ?? reset($raw) ?? null) : (string)$raw;

        if ($id === null) {
            return $this->renderDefault($raw, $meta);
        }

        return (string)($choices[$id] ?? $this->renderDefault($raw, $meta));
    }

    protected function renderMultiChoice(string|array $raw, array $meta): string
    {
        if (empty($raw)) {
            return '';
        }

        $choices = $meta['choices'] ?? [];
        $vals    = is_array($raw) ? $raw : [$raw];

        $labels = [];
        foreach ($vals as $v) {
            $key = is_array($v) ? ($v['id'] ?? null) : (string)$v;

            if ($key === null) {
                continue;
            }

            $labels[] = (string)($choices[$key] ?? $key);
        }

        return implode(', ', $labels);
    }

    protected function renderDefault(string|array $raw, array $meta): string
    {
        if (empty($raw)) {
            return '';
        }

        if (is_numeric($raw)) {
            return (string)$raw;
        }

        if (is_array($raw)) {
            $flat = [];
            array_walk_recursive($raw, function ($v) use (&$flat): void {
                $flat[] = is_scalar($v) ? (string)$v : json_encode($v, JSON_UNESCAPED_UNICODE);
            });

            return implode(', ', $flat);
        }

        return (string)$raw;
    }

    protected function mapStatus(?string $status): string
    {
        $status = $status ? strtolower($status) : null;
        return $this->statusMap[$status] ?? ($status ?? '');
    }

    protected function resolveRequester(Content $submission): string
    {
        if ($this->isAnonymous) {
            return 'Anonyme';
        }

        $payload  = $this->decodeOutput($submission->output);
        $userUuid = $payload['user_uuid'] ?? $submission->author_id ?? $submission->author ?? null;

        if (is_string($userUuid) && $userUuid !== '') {
            $user = User::query()->where('uuid', $userUuid)->first();

            if ($user) {
                $name = trim(($user->firstname ?? '') . ' ' . ($user->lastname ?? ''));

                if ($name !== '') {
                    return $name;
                }

                if (!empty($user->name)) {
                    return (string)$user->name;
                }

                if (!empty($user->email)) {
                    return (string)$user->email;
                }

                return (string)$userUuid;
            }
        }

        return '';
    }

    protected function mapUserInfos(Content $submission): array
    {
        if (empty($this->userInfoKeys)) {
            return [];
        }

        if ($this->isAnonymous) {
            return array_fill(0, count($this->userInfoKeys), '');
        }

        $payload  = $this->decodeOutput($submission->output);
        $userUuid = $payload['user_uuid'] ?? $submission->author_id ?? $submission->author ?? null;

        if (is_string($userUuid) && !empty($userUuid)) {
            $user = User::query()->where('uuid', $userUuid)->first();

            return array_map(function (string $key) use ($user): string {
                if ($user && isset($user->{$key})) {
                    return (string)$user->{$key};
                }
                return '';
            }, $this->userInfoKeys);
        }

        return array_fill(0, count($this->userInfoKeys), '');
    }

    protected function renderDate(string|array $date): string
    {
        if (!is_string($date)) {
            return '';
        }

        try {
            return get_date_in_user_timezone($date)?->format('d/m/Y H:i') ?? "";
        } catch (\Throwable $throwable) {
            return $date;
        }
    }
}
