<?php

use Illuminate\Support\Str;
use Inside\Host\Bridge\BridgeContent;

/**
 * @param mixed $user
 * @param BridgeContent $bridge
 * @param string $attribute
 * @param string $contentType
 * @return string|null
 * @throws Exception
 */
function getEntityReference($user, BridgeContent $bridge, string $attribute, string $contentType): ?string
{
    $title = $user->{$attribute};

    if (!$title) {
        return null;
    }

    $entity = call_user_func(type_to_class($contentType).'::query')
        ->where('title', $title)->first();

    if ($entity) {
        $uuid = $entity->uuid;
    } else {
        $uuid = $bridge->contentInsert($contentType, [
            'type' => 'node',
            'bundle' => $contentType,
            'title' => $title,
        ]);
    }

    return $uuid;
}

return [
    /**
     * Apply a post filter on users
     */
    'postfilter' => function ($users) {
        return $users->reject(function ($user) {
            return Str::lower($user->onPremisesExtensionAttributes->extensionAttribute14) !== "oui";
        });
    },

    'sync_attributes' => [
        'mail' => 'mail',
        'name' => 'mail',
        'langcode' => function ($user, BridgeContent $bridge) {
            return 'fr';
        },
        'status' => function ($user, BridgeContent $bridge) {
            return 1;
        },
        'firstname' => function ($user, BridgeContent $bridge) {
            return Str::ucfirst(Str::lower($user->givenName));
        },
        'lastname' => 'surname',
        'function' => 'jobTitle',
        'phone' => function ($user, BridgeContent $bridge) {
            $phone = implode(", ", $user->businessPhones);
            if (empty($phone)) {
                return " ";
            }
            return $phone;
        },
        'mobile' => 'mobilePhone',
        'managements' => function ($user, BridgeContent $bridge) {
            return getEntityReference($user, $bridge, 'department', 'managements');
        },
        'sub_managements' => function ($user, BridgeContent $bridge) {
            return $user->onPremisesExtensionAttributes->extensionAttribute7;
        },
        'services' => function ($user, BridgeContent $bridge) {
            return $user->onPremisesExtensionAttributes->extensionAttribute8;
        },
        'locations' => function ($user, BridgeContent $bridge) {
            return getEntityReference($user, $bridge, 'officeLocation', 'locations');
        },
        'manager' => function ($user, BridgeContent $bridge) {
            if (!isset($user->manager)) {
                return null;
            }

            if (!$user->manager->mail) {
                return null;
            }

            $inside_user = \Inside\Content\Models\Contents\Users::query()->where('email', $user->manager->mail)->first();

            if (!$inside_user) {
                return null;
            }
            return $inside_user->uuid;
        },
        'upn' => 'userPrincipalName',
    ],
];
