<?php

use Illuminate\Database\Migrations\Migration;
use Illuminate\Support\Facades\Artisan;
use Illuminate\Support\Facades\DB;
use Inside\Content\Facades\Schema as InsideSchema;
use Inside\Content\Models\Contents\ToolsCategories;
use Inside\Host\Bridge\BridgeContent;
use Inside\Host\Bridge\BridgeContentType;
use Inside\Host\Migrations\CreateContentTypes;
use Inside\Slug\Models\Slug;
use Inside\Content\Models\Contents\Tools;

class InstallToolsV2ForIimu extends Migration
{
    use CreateContentTypes;

    public function up(): void
    {
        $this->bootUp(__FILE__);

        $bridge = new BridgeContentType();
        $bridgeContent = new BridgeContent();

        if (! InsideSchema::hasContentType('tools')) {
            return;
        }

        if (! InsideSchema::hasContentType('tools_categories')) {
            $this->migrateContent('tools_categories');

            $bridge->contentTypeCreateOrUpdateField(
                type: 'tools',
                fieldName: 'tools_categories',
                options: [
                    'name' => 'tools_categories',
                    'searchable' => true,
                    'translatable' => true,
                    'search_result_field' => false,
                    'searchable_filter' => false,
                    'required' => true,
                    'weight' => -2,
                    'title' => [
                        'fr' => 'Catégorie d\'outils',
                        'en' => 'Tools category',
                        'es' => 'Categoría de herramientas',
                        'pt' => 'Categoria de ferramentas',
                        'de' => 'Werkzeugkategorie',
                        'nl' => 'Gereedschapscategorie',
                    ],
                    'widget' => [
                        'type' => BridgeContentType::SELECT_WIDGET,
                        'settings' => [],
                        'weight' => 1,
                    ],
                    'type' => BridgeContentType::REFERENCE_FIELD,
                    'settings' => [
                        'target_type' => ['tools_categories'],
                        'cardinality' => 1,
                    ],
                ],
                weight: -2
            );

            $defaultToolsCategories = $bridgeContent->contentInsert(
                type: 'tools_categories',
                data: [
                    'title' => 'Outils',
                    'langcode' => config('app.locale', 'fr'),
                ]
            );

            Tools::all()->each(function ($tool) use ($bridgeContent, $defaultToolsCategories) {
                $toolTitle = $tool->title ?? 'Outil';

                $bridgeContent->contentInsert(
                    type: 'tools',
                    data: [
                        'uuid' => $tool->uuid,
                        'tools_categories' => $defaultToolsCategories,
                        'title' => $toolTitle,
                    ]
                );
            });
        }

        if (! InsideSchema::hasContentType('tools_categories_listings')) {
            $this->migrateContent('tools_categories_listings');
            Artisan::call('models:rebuild');
        }

        $bridge->contentTypeCreateOrUpdateField(
            type: 'tools_categories',
            fieldName: 'tools_categories_listings',
            options: [
                'name' => 'tools_categories_listings',
                'searchable' => true,
                'translatable' => true,
                'search_result_field' => false,
                'searchable_filter' => false,
                'required' => true,
                'weight' => -2,
                'title' => [
                    'fr' => 'Listing catégorie d\'outils',
                    'en' => 'Tools categories listing',
                    'es' => 'Listado de categorías de herramientas',
                    'pt' => 'Listagem de categorias de ferramentas',
                    'de' => 'Werkzeugkategorienliste',
                    'nl' => 'Lijst van gereedschapscategorieën',
                ],
                'widget' => [
                    'type' => BridgeContentType::SELECT_WIDGET,
                    'settings' => [],
                    'weight' => 1,
                ],
                'type' => BridgeContentType::REFERENCE_FIELD,
                'settings' => [
                    'target_type' => ['tools_categories_listings'],
                    'cardinality' => 1,
                ],
            ],
            weight: -2
        );

        $toolsLang = [
            'fr' => 'Outils',
            'en' => 'Tools',
            'es' => 'Herramientas',
            'pt' => 'Ferramentas',
            'de' => 'Werkzeuge',
            'nl' => 'Gereedschap',
        ];

        $langs = array_unique(array_merge([config('app.locale', 'fr')], list_languages()));

        foreach ($langs as $lang) {
            Slug::where('slug', strtolower($toolsLang[$lang]) . '_old')
                ->delete();

            Slug::where('slug', strtolower($toolsLang[$lang]))
                ->update(['slug' => strtolower($toolsLang[$lang]) . '_old']);

            $uuidHost = DB::table('inside_content_tools_categories_listings')
                ->where('langcode', config('app.locale', 'fr'))
                ->first()
                ->uuid_host;

            $data = [
                'title' => $toolsLang[$lang],
                'langcode' => $lang,
            ];

            if ($uuidHost) {
                $data['uuid_host'] = $uuidHost;
            }

            $toolsCategoriesListings = $bridgeContent->contentInsert(
                type: 'tools_categories_listings',
                data: $data
            );

            $toolsCategories = ToolsCategories::where('langcode', $lang)
                ->pluck('uuid', 'title')
                ->each(function ($uuid, $title) use ($bridgeContent, $toolsCategoriesListings) {
                    $bridgeContent->contentInsert(
                        type: 'tools_categories',
                        data: [
                            'title' => $title,
                            'uuid' => $uuid,
                            'tools_categories_listings' => $toolsCategoriesListings,
                        ]
                    );
                });
        }

        Artisan::call('models:rebuild');
        $this->finishUp();
    }
}
