<?php

use Doctrine\DBAL\Events;
use Illuminate\Database\Migrations\Migration;
use Inside\Content\Events\CommentCreatedEvent;
use Inside\Content\Events\ContentCreatedEvent;
use Inside\Content\Models\Contents\Comments;
use Inside\Content\Models\Contents\Documents;
use Inside\Content\Models\Contents\News;
use Inside\Content\Models\Contents\Users;
use Inside\Notify\Models\NotificationSubscriber;
use Inside\Notify\Models\NotificationType;
use Inside\Notify\Models\NotificationTypeRole;
use Inside\Permission\Models\Role;
use Inside\Reaction\Events\ReactionSubscribedEvent;
use Symfony\Component\Console\Output\ConsoleOutput;

class NotificationTypesMigration extends Migration
{
    /**
     * @var ConsoleOutput
     */
    protected $console;

    public function __construct()
    {
        $this->console = new ConsoleOutput();
    }

    public function up(): void
    {
        $this->console->writeln('<info>Création des types de notifications</info>');
        $types = $this->getNotificationTypes();
        $this->console->writeln('<info>'.count($types).' types de notifications créés.</info>');

        $roles = Role::all();
        $users = Users::all();

        foreach ($types as $type) {
            if ($type->role) {
                foreach ($roles as $role) {
                    NotificationTypeRole::firstOrCreate(
                        [
                            'role_id' => $role->id,
                            'notification_type_id' => $type->id,
                        ]
                    );
                }
            }

            foreach ($users as $user) {
                NotificationSubscriber::firstOrCreate(
                    [
                        'user_uuid' => $user->uuid,
                        'notification_type_id' => $type->id,
                    ]
                );
            }
        }

        $this->console->writeln('<info>'.count($users).' Users subscribed.</info>');
        $this->console->writeln('<info>'.count($roles).' Roles subscribed.</info>');
    }

    public function down(): void
    {
        $this->console->writeln('<info>Suppression des types de notifications</info>');
        $types = $this->getNotificationTypes();
        $this->console->writeln('<info>'.count($types).' types de notifications supprimés.</info>');
    }

    public function getNotificationTypes(): array
    {
        $notificationTypes = [];
        // Global notifications
        // Someone commented a user content
        $notificationTypes[] = NotificationType::firstOrCreate(
            [
                'via' => 'web',
                'default' => true,
                'event' => CommentCreatedEvent::class,
                'model' => Comments::class,
                'action' => 'contentCommented',
                'type' => 'global',
                'condition' => 'parent_type:news,events|author:CURRENT_USER',
                'multiple' => true,
                'language' => false,
                'profile' => true,
            ],
            [
                'data' => [
                    'title' => 'notifications.create.comment.content.title',
                    'description' => 'notifications.create.comment.content.description',
                    'icon' => 'comment',
                    'text' => 'notifications.create.comment.content.<content_type>.text',
                    'fields' => [
                        'title',
                        'content_type',
                    ],
                ],
            ]
        );

        // Someone answer someone else comment
        $notificationTypes[] = NotificationType::firstOrCreate(
            [
                'via' => 'web',
                'default' => true,
                'event' => CommentCreatedEvent::class,
                'model' => Comments::class,
                'action' => 'commentCommented',
                'type' => 'global',
                'condition' => 'author:CURRENT_USER',
                'multiple' => true,
                'language' => false,
                'profile' => true,
            ],
            [
                'data' => [
                    'title' => 'notifications.create.comment.comment.title',
                    'description' => 'notifications.create.comment.comment.description',
                    'icon' => 'comment',
                    'text' => 'notifications.create.comment.comment.<content_type>.text',
                    'fields' => [
                        'title',
                        'content_type',
                    ],
                    'url' => 'ROOT_RELATION:news,events,documents',
                ],
            ]
        );

        // News is featured
        $notificationTypes[] = NotificationType::firstOrCreate(
            [
                'via' => 'web',
                'default' => true,
                'event' => ContentCreatedEvent::class,
                'model' => News::class,
                'action' => 'createOrUpdate',
                'condition' => 'is_featured:1|status:1',
                'multiple' => false,
                'language' => true,
                'profile' => true,
                'role' => true,
            ],
            [
                'data' => [
                    'title' => 'notifications.create.news.featured.title',
                    'description' => 'notifications.create.news.featured.description',
                    'icon' => 'news',
                    'text' => 'notifications.create.news.featured.text',
                    'fields' => [
                        'title',
                    ],
                ],
            ]
        );

        // Someone liked a content/comment
        $notificationTypes[] = NotificationType::firstOrCreate(
            [
                'via' => 'web',
                'default' => true,
                'event' => ReactionSubscribedEvent::class,
                'model' => null,
                'action' => 'like',
                'condition' => 'author:CURRENT_USER',
                'multiple' => true,
                'language' => false,
                'profile' => true,
            ],
            [
                'data' => [
                    'title' => 'notifications.reaction.like.title',
                    'description' => 'notifications.reaction.like.description',
                    'icon' => 'like',
                    'text' => 'notifications.reaction.like.<content_type>.text',
                    'fields' => [
                        'content_type',
                        'title',
                    ],
                ],
            ]
        );

        // Activity notifications
        // Prénom Nom a commenté l'actualité titre de l'actualité
        $notificationTypes[] = NotificationType::firstOrCreate(
            [
                'via' => 'web',
                'default' => true,
                'event' => ContentCreatedEvent::class,
                'model' => Comments::class,
                'action' => 'contentCommented',
                'type' => 'activity',
                'condition' => 'parent_type:news',
                'multiple' => false,
                'language' => true,
                'profile' => false,
                'role' => true,
            ],
            [
                'data' => [
                    'title' => 'notifications.activity.create.comment.news.title',
                    'description' => 'notifications.activity.create.comment.news.description',
                    'text' => 'notifications.activity.create.comment.news.text',
                    'fields' => ['title'],
                    'url' => 'RELATION:news',
                ],
            ]
        );

        // Prénom Nom a commenté l'évènement titre de l'évènement
        $notificationTypes[] = NotificationType::firstOrCreate(
            [
                'via' => 'web',
                'default' => true,
                'event' => ContentCreatedEvent::class,
                'model' => Comments::class,
                'action' => 'contentCommented',
                'type' => 'activity',
                'condition' => 'parent_type:events',
                'multiple' => false,
                'language' => true,
                'profile' => false,
                'role' => true,
            ],
            [
                'data' => [
                    'title' => 'notifications.activity.create.comment.events.title',
                    'description' => 'notifications.activity.create.comment.events.description',
                    'text' => 'notifications.activity.create.comment.events.text',
                    'fields' => ['title'],
                    'url' => 'RELATION:events',
                ],
            ]
        );

        // Prénom Nom a ajouté une ressource
        $notificationTypes[] = NotificationType::firstOrCreate(
            [
                'via' => 'web',
                'default' => true,
                'event' => ContentCreatedEvent::class,
                'model' => Documents::class,
                'action' => 'create',
                'type' => 'activity',
                'condition' => '',
                'multiple' => false,
                'language' => true,
                'profile' => false,
                'role' => true,
            ],
            [
                'data' => [
                    'title' => 'notifications.activity.create.documents.title',
                    'description' => 'notifications.activity.create.documents.description',
                    'text' => 'notifications.activity.create.documents.text',
                    'fields' => [
                        'title',
                        'slug',
                    ],
                    'url' => 'GED:folders',
                ],
            ]
        );

        // Prénom Nom a ajouté un évènement
        $notificationTypes[] = NotificationType::firstOrCreate(
            [
                'via' => 'web',
                'default' => true,
                'event' => ContentCreatedEvent::class,
                'model' => Events::class,
                'action' => 'create',
                'type' => 'activity',
                'condition' => '',
                'multiple' => false,
                'language' => true,
                'profile' => false,
                'role' => true,
            ],
            [
                'data' => [
                    'title' => 'notifications.activity.create.events.title',
                    'description' => 'notifications.activity.create.events.description',
                    'text' => 'notifications.activity.create.events.text',
                    'fields' => [
                        'title',
                        'slug',
                    ],
                ],
            ]
        );

        // Prénom Nom a ajouté l'actualité
        $notificationTypes[] = NotificationType::firstOrCreate(
            [
                'via' => 'web',
                'default' => true,
                'event' => ContentCreatedEvent::class,
                'model' => News::class,
                'action' => 'create',
                'type' => 'activity',
                'condition' => '',
                'multiple' => false,
                'language' => true,
                'profile' => false,
                'role' => true,
            ],
            [
                'data' => [
                    'title' => 'notifications.activity.create.news.title',
                    'description' => 'notifications.activity.create.news.description',
                    'text' => 'notifications.activity.create.news.text',
                    'fields' => [
                        'title',
                        'slug',
                    ],
                ],
            ]
        );

        return $notificationTypes;
    }
}
