<?php

namespace Inside\ICHB\Services;

use Carbon\Carbon;
use Carbon\CarbonPeriod;
use Exception;
use Illuminate\Support\Collection;
use Illuminate\Database\Eloquent\Collection as DatabaseCollection;
use Tightenco\Collect\Support\Collection as TightencoCollection;
use Inside\Content\Models\Contents\DutiesServices;
use Inside\Permission\Facades\Permission;
use Inside\Support\Str;

/**
 * Class Duty
 *
 * @package Inside\ICHB\src\Services
 */
class Duty
{
    /**
     * get Duties list by days by services from $startDate to $endDate
     *
     * @param string $startDate
     * @param string $endDate
     * @return Collection|TightencoCollection
     */
    public function getDutiesList(string $startDate, string $endDate): Collection|TightencoCollection
    {
        // Convert to carbon
        try {
            $startDate = Carbon::createFromTimestamp((int)$startDate, 'Europe/Paris');
            $startDate->setTime(12, 0, 0);
            $endDate = Carbon::createFromTimestamp((int)$endDate, 'Europe/Paris');
            $endDate->setTime(12, 0, 0);
        } catch (Exception $e) {
            throw new \InvalidArgumentException('start or end date is invalid');
        }

        $services = DutiesServices::whereStatus(true)->orderBy('title')->get();
        $dutiesByServices = collect();
        $headers = null;
        foreach ($services as $service) {
            $dutiesByService = $this->getDutiesByDayForService($startDate, $endDate, $service);
            if ($headers === null) {
                // Set headers
                $headers = collect(
                    $dutiesByService->keys()->transform(
                        function ($date) {
                            $date = Carbon::createFromFormat('Y-m-d', $date);

                            if (!$date instanceof Carbon) {
                                return '';
                            }

                            return Str::upper($date->isoFormat('dddd D'));
                        }
                    )
                );
            }
            $dutiesByServices[] = [
                'uuid' => $service->uuid,
                'title' => $service->title,
                'slug' => $service->slug[0] ?? null,
                'admin' => [
                    'permission' => [
                        'update' => Permission::allowed('update', 'duties_services', $service->uuid),
                        'delete' => Permission::allowed('delete', 'duties_services', $service->uuid),
                    ],
                ],
                'days' => $dutiesByService,
            ];
        }

        return collect(
            [
                'headers' => $headers,
                'services' => $dutiesByServices,
            ]
        );
    }

    /**
     * Get a collection of duties by days for $service from $startDate to $endDate
     */
    protected function getDutiesByDayForService(Carbon $startDate, Carbon $endDate, DutiesServices $service): Collection|TightencoCollection
    {
        $days = collect();
        $table = type_to_table('duties');

        //  Get all services from $startDate tp $endDate
        $period = CarbonPeriod::create($startDate, $endDate);
        foreach ($period as $day) {
            if ($day === null) {
                continue;
            }

            $duties = $this->getDutiesForDayForService($day->toMutable(), $service);
            $result = collect();
            foreach ($duties as $duty) {
                $dutyStartDate = Carbon::createFromTimestamp($duty->date)->setTime(0, 0, 0);
                $dutyEndDate = Carbon::createFromTimestamp($duty->end_date)->setTime(0, 0, 0);
                // Prepare result
                $result[] = [
                    'uuid' => $duty->uuid,
                    'title' => $duty->title,
                    'admin' => [
                        'permission' => [
                            'update' => Permission::allowed('update', 'duties', $duty->uuid),
                            'delete' => Permission::allowed('delete', 'duties', $duty->uuid),
                        ],
                    ],
                    'start_date' => $dutyStartDate->format('Y-m-d'),
                    'end_date' => $dutyEndDate->format('Y-m-d'),
                    'time' => $duty->time,
                    'weight' => $duty->weight ?? 0,
                    'duty_user' => [
                        'uuid' => $duty->dutyUser()->first()->uuid ?? null,
                        'firstname' => $duty->dutyUser()->first()->firstname ?? null,
                        'lastname' => $duty->dutyUser()->first()->lastname ?? null,
                    ],
                ];
            }
            $days[$day->format('Y-m-d')] = $result->sortBy('weight')->values();
        }

        return $days;
    }

    /**
     * get Duties for $day for $service
     */
    protected function getDutiesForDayForService(Carbon $day, DutiesServices $service): DatabaseCollection
    {
        $table = type_to_table('duties');
        $startDay = clone $day;
        $startDay->setTime(0, 0, 0);
        $endDay = clone $day;
        $endDay->setTime(23, 59, 59);

        return $service->reverseDuties()->where($table.'.status', true)->where(
            function ($query) use ($startDay, $endDay, $table) {
                $query->whereBetween(
                    $table.'.date',
                    [$startDay, $endDay]
                )->orWhereBetween($table.'.end_date', [$startDay, $endDay])->orWhere(
                    function ($query) use ($startDay, $endDay, $table) {
                        $query->whereDate($table.'.date', '<=', $startDay)->whereDate(
                            $table.'.end_date',
                            '>=',
                            $endDay
                        );
                    }
                );
            }
        )->get();
    }
}
