<?php

use Adldap\Laravel\Facades\Adldap;
use Adldap\Models\Entry;
use Adldap\Models\Model as AdldapModel;
use Illuminate\Console\Command;
use Illuminate\Support\Collection;
use Tightenco\Collect\Support\Collection as TightencoCollection;
use Illuminate\Support\Facades\Log;
use Inside\Authentication\Models\User;
use Inside\Content\Models\Contents\Company;
use Inside\Content\Models\Contents\Functions;
use Inside\Content\Models\Contents\OrganisationalServices;
use Inside\Content\Models\Contents\Services;
use Inside\Content\Models\Contents\Subservices;
use Inside\Content\Models\Contents\Users;
use Inside\Content\Models\Contents\UsersTitles;
use Inside\Host\Bridge\BridgeContent;
use Inside\Support\Str;
use Symfony\Component\Console\Output\OutputInterface;

/**
 * @param mixed $user
 * @param string $title
 * @return bool
 */
function ichbMatchTitle($user, string $title): bool
{
    foreach ($user->functions as $function) {
        if (Str::contains(Str::lower($function->title), $title)) {
            return true;
        }
    }

    return false;
}

/**
 * @param mixed $user
 * @return bool
 */
function ichbDetectDoctor($user): bool
{
    return ichbMatchTitle($user, 'chef de clinique')
        || ichbMatchTitle($user, 'assistant(e) spécialiste')
        || ichbMatchTitle($user, 'assistant spécialiste')
        || ichbMatchTitle($user, 'assistant(e) associé(e)')
        || ichbMatchTitle($user, 'assistant associé')
        || ichbMatchTitle($user, 'praticien')
        || ichbMatchTitle($user, 'assistant.Univ')
        || ichbMatchTitle($user, 'assistant(e) univ')
        || ichbMatchTitle($user, 'assistant univ')
        || ichbMatchTitle($user, 'consultant(e)')
        || ichbMatchTitle($user, 'consultant')
        || ichbMatchTitle($user, 'médecin')
        || ichbMatchTitle($user, 'radiologue');
}

/**
 * @param mixed $user
 * @return bool
 */
function ichbDetectProfessor($user): bool
{
    return ichbMatchTitle($user, 'professeur');
}

function ichbWriteResult(OutputInterface $console, bool $success): void
{
    if ($success) {
        $console->writeln(' <fg=green>✔</fg=green>');

        return;
    }
    $console->writeln(' <fg=red>✘</fg=red>');
}

function ichbSeparatorLine(OutputInterface $console, ?string $color = null): void
{
    $colorOn = $color !== null ? '<fg='.$color.'>' : '';
    $colorOff = $color !== null ? '</fg='.$color.'>' : '';
    $console->writeln($colorOn.str_repeat('-', 80).$colorOff);
}

function create_new_service(
    Collection|TightencoCollection $organisationsPath,
    OutputInterface $console,
    BridgeContent $bridge
): bool {
    Log::info('[postFlight] OrganisationalServices <'.$organisationsPath->implode('/').'> Non trouvé => création');
    $console->writeln('<fg=magenta>Service <fg=cyan>'.$organisationsPath->implode('/').'</fg=cyan> inexistant</fg=magenta>');
    $parentUuid = null;
    $code = collect();
    $organisations = $organisationsPath;
    // phy/dmn (  dmn<1> / phy<2> => ok)
    foreach ($organisationsPath->reverse() as $organisation) {
        $code->prepend($organisation);
        Log::info('[postFlight] OrganisationalServices Recherche <'.$code->implode('/').'>');
        $service = OrganisationalServices::where('code', $code->implode('/'))->first();
        if (!$service) {
            Log::info('[postFlight] OrganisationalServices <'.collect($organisations)->implode('/').'> Non trouvé');
            // Get service name from ldap
            $organizationalUnit = Adldap::search()->findBy('OU', $organisation);

            if (!$organizationalUnit instanceof AdldapModel) {
                $console->writeln('<fg=magenta>OU <fg=cyan>'.$organisation.'</fg=cyan> introuvable</fg=magenta>');
                Log::info('[postFlight] OrganisationalServices OU introuvable <'.$organisation.'> Non trouvé');

                return false;
            }
            $description = null;
            if ($organizationalUnit->hasAttribute('description')) {
                $description = $organizationalUnit->getFirstAttribute('description');
                Log::info('[postFlight] Tentative de création OrganisationalServices <'.$description.'> ('.$code->implode('/').')');
                if (!empty($description)) {
                    $console->write(
                        'Création du service <fg=cyan>'.$description
                        .'</fg=cyan> <fg=green>'.$code->implode('/')
                        .'</fg=green>'
                    );
                    $parentUuid = $bridge->contentInsert('organisational_services', [
                        'title' => $description,
                        'code' => $code->implode('/'),
                        'organisational_services' => $parentUuid,
                    ]);
                    ichbWriteResult($console, $parentUuid !== null);
                } else {
                    return false;
                }
            } else {
                return false;
            }
        }
    }

    return true;
}

return [
    'user_filter' => 'is_from_ldap',
    'dont_check_email' => true,
    'usernames' => [
        'ldap' => [
            'discover' => env('ADLDAP_LDAP_DISCOVER', 'mail'),
            'authenticate' => env('ADLDAP_LDAP_AUTHENTICATE', 'mail'),
        ],
        'eloquent' => env('ADLDAP_ELOQUENT_VALUE', 'email'),
    ],
    'discovering_attribute' => 'userprincipalname',
    'sync_attributes' => [
        // Inside => LDAP
        'name' => 'samaccountname',
        'mail' => 'mail',
        'firstname' => function (Entry $user, BridgeContent $bridge) {
            if (!$user->hasAttribute('givenname')) {
                return null;
            }

            return Str::ucfirst(Str::lower($user->getFirstAttribute('givenname')));
        },
        'password' => function (Entry $user, BridgeContent $bridge) {
            return Str::random(64);
        },
        'lastname' => function (Entry $user, BridgeContent $bridge) {
            if (!$user->hasAttribute('sn')) {
                return null;
            }

            return Str::upper($user->getFirstAttribute('sn'));
        },
        'cn' => 'cn',
        'dn' => 'distinguishedname',
        'phone' => function (Entry $user, BridgeContent $bridge) {
            if (!$user->hasAttribute('mail')) {
                return null;
            }

            $userInside = Users::where('email', $user->getFirstAttribute('mail'))->first();

            if ($userInside === null || $userInside->phone === null) {
                $telephoneNumber = $user->getFirstAttribute('telephonenumber');

                if (! $telephoneNumber) {
                    return null;
                }

                if (str_contains($telephoneNumber, '/')) {
                    $telephoneNumber = explode('/', $telephoneNumber)[0];
                }

                return strlen($telephoneNumber) === 4 || strlen($telephoneNumber) === 10
                ? $telephoneNumber
                : null;
            }

            return $userInside->phone;
        },
        'second_phone'  => function (Entry $user, BridgeContent $bridge) {
            if (!$user->hasAttribute('mail')) {
                return null;
            }

            $userInside = Users::where('email', $user->getFirstAttribute('mail'))->first();
            $telephoneNumber = $user->getFirstAttribute('telephonenumber');

            if (($userInside === null || $userInside->second_phone === null) && $telephoneNumber && str_contains($telephoneNumber, '/')) {
                $telephoneNumber = explode('/', $telephoneNumber)[1];

                return strlen($telephoneNumber) === 4 || strlen($telephoneNumber) === 10
                    ? $telephoneNumber
                    : null;
            }

            return $userInside->second_phone;
        },
        'status' => function (Entry $user, BridgeContent $bridge) {
            return false; // User will be enable in postfly if he is in, at least, one service
        },
        'functions' => function (Entry $user, BridgeContent $bridge) {
            if (!$user->hasAttribute('title')) {
                return null;
            }

            $functions = [];
            $userFunctions = $user->getAttribute('title');
            foreach ($userFunctions as $functionName) {
                if (empty($functionName)) {
                    continue;
                }
                $uuid = false;
                $function = Functions::where('title', $functionName)->first();

                if ($function) {
                    $uuid = $function->uuid;
                } else {
                    $uuidBridge = $bridge->contentInsert('functions', [
                        'type' => 'node',
                        'bundle' => 'functions',
                        'title' => $functionName,
                        'uid' => 1,
                    ]);

                    $function = Functions::find($uuidBridge);

                    if ($function) {
                        $uuid = $function->uuid;
                    }
                }

                if ($uuid) {
                    $functions[] = $uuid;
                }
            }

            if (!empty($functions)) {
                return array_first($functions);
            }

            return null;
        },
        'location' => function (Entry $user, BridgeContent $bridge) {
            return '';
        },
        'services' => function (Entry $user, BridgeContent $bridge) {
            return []; // Set in postfly
        },
        'is_from_ldap' => function (Entry $user, BridgeContent $bridge) {
            return true;
        },
    ],
    /**
     * @var  Command $console
     * @var array $imported
     * @var BridgeContent $bridge
     */
    'postflight' => function ($command, $imported, $bridge) {
        $console = $command->getOutput();
        $console->writeln('<fg=white><bg=cyan>Début de l\'assignation des services</fg=white></bg=cyan>');
        foreach ($imported as $userUuid) {
            /** @var User|null $user */
            $user = User::find($userUuid);
            if ($user === null) {
                continue;
            }
            $user = $user->information;
            if ($user === null || $user->dn === null || $user->cn === null) {
                continue;
            }
            $console->writeln(
                'Assignation du service de <fg=yellow>'.$user->cn
                .'</fg=yellow>'
            );
            Log::info('[postFlight] Assigning services to user <'.$user->cn.'>');
            $organisationStr = trim(
                Str::before(
                    Str::after(
                        Str::lower($user->dn),
                        'cn='.Str::lower($user->cn)
                    ),
                    Str::lower(env('ADLDAP_BASE_DN'))
                ),
                ','
            );
            Log::info('[postFlight] Detected OU <'.$organisationStr.'>');
            if (!empty($organisationStr)) {
                /** @var array $organisations */
                $organisations = explode(',', $organisationStr);
                if (empty($organisations)) {
                    continue;
                }
                $organisations = collect($organisations)->map(function (
                    $organisation
                ) {
                    return Str::after($organisation, 'ou=');
                });
                // phy,dmn ( dmn<1> / phy<2> => reversed)
                if ($organisations->isEmpty()) {
                    $organisations[] = 'dmn';
                }
                $organisationsPath = $organisations->reverse();// dmn,phy  (  dmn<1> / phy<2> => ok)
                $serviceOrganisations = $organisations;

                Log::info('[postFlight] Ajustement OrganisationalServices <'.$organisationsPath->implode('/').'>');
                $console->writeln('Service <fg=cyan>'.$organisationsPath->implode('/').'</fg=cyan>');

                //  example: "CN=DOE John,OU=PHY,OU=DMN,OU=DGE,OU=Utilisateurs,DC=chb,DC=local",
                // dmn/phy (  dmn<1> / phy<2> => ok)
                $service = OrganisationalServices::where('code', $organisationsPath->implode('/'))->first();
                if (!$service) {
                    if (!create_new_service($organisationsPath, $console, $bridge)) {
                        Log::info('[postFlight] Impossible de créer un service <'.$organisationsPath->implode('/').'>');
                        continue;
                    }
                }

                $organisationService = [$service->uuid];
                $organisation = $serviceOrganisations->first();
                $organizationalUnit = Adldap::search()->findBy('OU', $organisation ?? '');
                $title = null;
                if ($organizationalUnit instanceof AdldapModel && $organizationalUnit->hasAttribute('st')
                ) {
                    $title = $organizationalUnit->getFirstAttribute('st');
                }
                $serviceUuids = [];
                if ($title !== null) {
                    Log::info('[postFlight] Service <'.$title.'>');

                    $service = Services::where('title', $title)->first();
                    if ($service === null) {
                        Log::info('[postFlight] création Service <'.$title.'>');
                        $console->write('Création du service hospitalier <fg=cyan>'.$title.'</fg=cyan>');
                        $serviceUuid = $bridge->contentInsert('services', [
                            'title' => $title,
                        ]);
                        ichbWriteResult($console, $serviceUuid !== null);
                        $serviceUuids[] = $serviceUuid;
                    } else {
                        $serviceUuids[] = $service->uuid;
                    }
                }

                // handle subservices
                $subserviceUuid = null;
                if ($organizationalUnit instanceof AdldapModel && $organizationalUnit->hasAttribute('l')) {
                    $subserviceTitle = $organizationalUnit->getFirstAttribute('l');
                    if ($subserviceTitle !== null) {
                        $subservice = Subservices::where('title', $subserviceTitle)->first();
                        if ($subservice === null) {
                            Log::info('[postFlight] création sous-Service <'.$subserviceTitle.'>');
                            $console->write('Création du sous-service hospitalier <fg=cyan>'.$subserviceTitle.'</fg=cyan>');
                            $subserviceUuid = $bridge->contentInsert('subservices', [
                                'title' => $subserviceTitle,
                                'bundle' => 'subservices',
                                'type' => 'node',
                            ]);
                            ichbWriteResult($console, $subserviceUuid !== null);
                        } else {
                            $subserviceUuid = $subservice->uuid;
                        }
                    }
                }
            } else {
                // Not main service from AD
                $serviceUuids = [];
            }

            // Add other_services
            $otherServiceUuids = $user->otherServices->isNotEmpty() ? $user->otherServices->pluck('uuid')->toArray() : null;
            if ($otherServiceUuids !== null) {
                Log::info('[postFlight] Ajout services supplémentaires <'.$user->otherServices->pluck('title')->implode(', ').'>');
                $console->write('Ajout services supplémentaires <fg=cyan>'.$user->otherServices->pluck('title')->implode(', ').'</fg=cyan>');
                $serviceUuids = array_merge($serviceUuids, $otherServiceUuids);
            }

            // Prepare data
            $data = [
                'uuid' => $userUuid,
                'organisational_services' => $organisationService ?? null,
                'services' => $serviceUuids,
                'subservices' => $subserviceUuid ?? null,
                'status' => !empty($serviceUuids),
            ];

            // Detect title
            if (($user->usersTitles === null || $user->usersTitles->isEmpty())
            ) {
                if (ichbDetectDoctor($user)) {
                    if (($userTitle = UsersTitles::where('title', 'Dr')->first()) !== null
                    ) {
                        $data['users_titles'] = $userTitle->uuid;
                    }
                } elseif (ichbDetectProfessor($user)) {
                    if (($userTitle = UsersTitles::where('title', 'Pr')->first()) !== null
                    ) {
                        $data['users_titles'] = $userTitle->uuid;
                    }
                }
            }

            Log::info('[postFlight] Maj utilisateur');
            $console->writeln('Mise-à-jour des informations');
            Log::info('[postFlight] setting <'.json_encode($data).'>');
            $bridge->contentUpdate('users', $data);
            ichbSeparatorLine($console, 'green');
        }

        $console->writeln('<fg=white;bg=cyan>Activation des services non vide</fg=white;bg=cyan>');
        Services::has('reverseUsers')->each(function ($service) use ($bridge) {
            $bridge->contentUpdate('services', [
                'uuid' => $service->uuid,
                'status' => true,
            ]);
        });

        $console->writeln('<fg=white;bg=cyan>Désactivation des services vide</fg=white;bg=cyan>');
        Services::doesntHave('reverseUsers')->each(function ($service) use ($bridge) {
            $bridge->contentUpdate('services', [
                'uuid' => $service->uuid,
                'status' => false,
            ]);
        });
    },
];
