import { http } from '@inside-library/helpers/Http'

const basicFields = [
  'uuid',
  'title',
  'slug',
  'content_type',
  'admin',
  'published_at'
]

const olympicNewsFilters = {
  limit: 3,
  sort: 'published_at:desc',
  news_categories: { 'title:eq': 'Olympiades' }
}

const olympicNewsFields = [
  ...basicFields,
  'chapo',
  'image',
  { news_categories: ['title', 'uuid'] },
  { reactions: ['count', { users: ['uuid'] }] },
  { comments: ['count'] }
]

const fetchOlympic = async ({ $route }) => {
  try {
    const { data } = await http.get('content/olympics', {
      filters: {
        'slug:eq': $route.params.slug
      },
      fields: [
        ...basicFields,
        'cta_text',
        'cta_link',
        'modal_name',
        'content',
        'introduction',
        'final_title',
        'final_date',
        'semifinal_title',
        'semifinal_date',
        { final_link: ['slug', 'uuid'] },
        { semifinal_link: ['slug', 'uuid'] }
      ]
    })

    const sequences = await http.get('content/sequences', {
      filters: {
        status: 1,
        sort: 'date:asc',
        olympics: {
          'slug:eq': $route.params.slug
        }
      },
      fields: [
        ...basicFields,
        'date',
        'end_date',
        'podium_1',
        'podium_2',
        'podium_3',
        'file',
        {
          sequences_status: {
            fields: ['title', 'color', 'name_machine']
          }
        }
      ]
    })

    const news = await http.get('content/news', {
      filters: olympicNewsFilters,
      fields: olympicNewsFields
    })

    return {
      ...data[0],
      sequences: sequences.data,
      news: news.data
    }
  } catch (err) {
    throw new Error(err)
  }
}

const fetchOlympicSemifinal = async ({ $route }) => {
  try {
    const { data } = await http.get('content/olympics_semifinal', {
      filters: {
        'slug:eq': $route.params.slug
      },
      fields: [
        ...basicFields,
        'introduction',
        'modal_title',
        'content',
        'semifinal_title',
        'semifinal_content',
        'button_text',
        'button_file',
        'show_final',
        'separator_content',
        'final_title',
        'final_content',
        'final_button_text',
        { olympics_final: ['uuid', 'slug'] },
        {
          olympics_teams: {
            fields: [
              'uuid',
              'title',
              'introduction',
              {
                users: {
                  fields: ['uuid', 'image'],
                  filters: { limit: 3 }
                }
              }
            ],
            filters: {
              'status:eq': 1,
              limit: 6
            }
          }
        },
        {
          final_teams: {
            fields: [
              'uuid',
              'title',
              'introduction',
              {
                users: {
                  fields: ['uuid', 'image'],
                  filters: { limit: 3 }
                }
              }
            ],
            filters: {
              'status:eq': 1,
              limit: 3
            }
          }
        },
        {
          olympics: {
            fields: ['uuid', 'slug'],
            filters: {
              semifinal_link: {
                'slug:eq': $route.params.slug
              },
              reverse: true
            }
          }
        }
      ]
    })

    const news = await http.get('content/news', {
      filters: olympicNewsFilters,
      fields: olympicNewsFields
    })

    return {
      ...data[0],
      news: news.data
    }
  } catch (err) {
    throw new Error(err)
  }
}

const fetchOlympicFinal = async ({ $route }) => {
  try {
    const { data } = await http.get('content/olympics_final', {
      filters: {
        'slug:eq': $route.params.slug
      },
      fields: [
        ...basicFields,
        'introduction',
        'final_title',
        'final_content',
        'final_desc',
        'trial_1',
        'trial_2',
        'trial_3',
        'button_text',
        'button_link',
        'content',
        'show_podium',
        {
          olympics_teams: {
            fields: ['title', 'uuid', 'image', 'introduction', 'weight'],
            filters: {
              'status:eq': 1,
              sort: 'weight:asc',
              limit: 3
            }
          }
        },
        {
          olympics: {
            fields: ['uuid', 'slug'],
            filters: {
              semifinal_link: {
                'slug:eq': $route.params.slug
              },
              reverse: true
            }
          }
        }
      ]
    })

    return {
      ...data[0]
    }
  } catch (err) {
    throw new Error(err)
  }
}

export { fetchOlympic, fetchOlympicSemifinal, fetchOlympicFinal }
