import { http } from '@inside-library/helpers/Http'

const fetchDescription = async ({ $route }) => {
  try {
    const { data } = await http.get('content/services_presentation', {
      filters: {
        sort: 'order:asc',
        services_space: {
          slug: $route.path.replace('/', '')
        },
        'status:eq': 1
      },
      fields: [
        'title',
        'summary',
        'uuid',
        'content_type',
        'admin',
        'link',
        'is_modale',
        'slug',
        'cta_label',
        'display_button'
      ]
    })

    return data
  } catch (err) {
    throw new Error(err)
  }
}

const fetchNews = async ({ $route }) => {
  try {
    const serviceData = await http.get('content/services_space', {
      filters: {
        slug: $route.path.replace('/', '')
      },
      fields: [{ news_categories: { fields: ['uuid'] } }]
    })

    const { data } = await http.get('content/news', {
      filters: {
        limit: 5,
        sort: 'published_at:desc',
        news_categories: {
          'uuid:in': serviceData.data[0].news_categories.data.map(
            category => category.uuid
          )
        },
        'status:eq': 1
      },
      fields: [
        'title',
        'uuid',
        'content_type',
        'chapo',
        'image',
        'slug',
        { reactions: ['count', { users: ['uuid'] }] }
      ]
    })

    return serviceData.data[0].news_categories.data[0] ? data : null
  } catch (err) {
    throw new Error(err)
  }
}

const fetchTabs = async ({ $route }) => {
  try {
    const serviceTabs = await http.get('content/services_tabs', {
      filters: {
        sort: 'order:asc',
        services_space: {
          slug: $route.path.replace('/', '')
        },
        'status:eq': 1
      },
      fields: ['title', 'uuid', 'admin', 'content_type']
    })

    const serviceContents = await http.get('content/services_content', {
      filters: {
        limit: 5,
        sort: 'order:asc',
        services_tabs: {
          'uuid:in': serviceTabs.data.map(tab => tab.uuid)
        },
        'status:eq': 1
      },
      fields: [
        'title',
        'uuid',
        'chapo',
        'slug',
        'services_content_link',
        'services_space',
        'is_modale',
        'display_as_cta',
        'image',
        'content_type',
        'admin',
        { services_tabs: { fields: ['uuid'] } }
      ]
    })

    return {
      tabs: serviceTabs.data,
      contents: serviceContents.data
    }
  } catch (err) {
    throw new Error(err)
  }
}

const fetchNeedsPage = async ({ $route }) => {
  try {
    const { data } = await http.get('content/needs_page', {
      filters: {
        sort: 'order:asc',
        services_space: {
          slug: $route.path.replace('/', '')
        },
        'status:eq': 1
      },
      fields: ['title', 'uuid', 'content_type', 'is_modale', 'link', 'slug']
    })

    return data
  } catch (err) {
    throw new Error(err)
  }
}

const fetchCTAServicesContent = async ({ $route }) => {
  try {
    const serviceContents = await http.get('content/services_content', {
      filters: {
        sort: 'order:asc',
        'status:eq': 1,
        services_space: { 'slug:eq': $route.path.replace(/^\//, '') },
        display_as_cta: true
      },
      fields: [
        'title',
        'uuid',
        'services_content_link',
        'services_space',
        'is_modale',
        'display_as_cta',
        'image',
        'admin',
        'content_type',
        'slug',
        { services_page_group: { fields: ['uuid'] } },
        { services_tabs: { fields: ['uuid'] } }
      ]
    })

    return serviceContents.data
  } catch (err) {
    throw new Error(err)
  }
}

const fetchPagesInGroups = async ({ $route }) => {
  try {
    const serviceGroups = await http.get('content/services_page_group', {
      filters: {
        sort: 'order:asc',
        services_space: {
          slug: $route.path.replace('/', '')
        },
        'status:eq': 1
      },
      fields: ['title', 'uuid', 'image', 'admin', 'content_type']
    })

    const serviceContents = await http.get('content/services_content', {
      filters: {
        sort: 'order:asc',
        services_page_group: {
          'uuid:in': serviceGroups.data.map(group => group.uuid)
        },
        'status:eq': 1
      },
      fields: [
        'title',
        'uuid',
        'slug',
        'services_content_link',
        'services_space',
        'is_modale',
        'display_as_cta',
        'admin',
        'content_type',
        { services_page_group: { fields: ['uuid'] } }
      ]
    })

    return {
      groups: serviceGroups.data,
      contents: serviceContents.data
    }
  } catch (err) {
    throw new Error(err)
  }
}

export {
  fetchDescription,
  fetchNews,
  fetchCTAServicesContent,
  fetchTabs,
  fetchNeedsPage,
  fetchPagesInGroups
}
