<?php

namespace Inside\HPSJ\Console;

use Illuminate\Console\Command;
use Illuminate\Support\Arr;
use Inside\Content\Models\Contents\Services;
use Inside\Content\Models\Contents\Users;
use Inside\Host\Bridge\BridgeContent;
use League\Csv\Reader;

class ServiceImporterCommand extends Command
{
    /**
     * @var string
     */
    protected $name = 'hpsj:services:import';

    /**
     * @var string
     */
    protected $signature = 'hpsj:services:import { --P|path=}';

    /**
     * @var string
     */
    protected $description = 'Import services';

    /**
     * @var string
     */
    protected $filePath = __DIR__ . '/../../../../../imports/services.csv';


    /**
     * Create a new command instance.
     *
     * @return void
     */
    public function __construct()
    {
        parent::__construct();
    }

    private function getReference(array $record, BridgeContent $bridge, string $bundle, string $column): ?array
    {
        $name = trim(ucwords(strtolower($record[$column])));

        if (empty($name)) {
            return null;
        }

        $query = call_user_func('Inside\\Content\\Models\\Contents\\' . \Illuminate\Support\Str::studly($bundle) . '::query');

        $entity = $query->where('title', $name)->first();

        if ($entity) {
            return [$entity->uuid];
        }
        $uuid = $bridge->contentInsert($bundle, [
            'title' => $name,
            'type' => 'node',
            'bundle' => $bundle
        ]);

        if ($uuid) {
            return [$uuid];
        }
        return null;
    }

    private function getUserReferences(array $record, string $column): array
    {
        $users = explode('/', $record[$column]);
        $uuids = [];

        foreach ($users as $user) {
            [$firstname, $lastname] = explode(' ', trim($user), 2);
            $query = Users::query()
                ->where('firstname', $firstname)
                ->where('lastname', $lastname)
                ->first();
            if ($query) {
                $uuids[] = $query->uuid;
            }
        }
        return $uuids;
    }

    private function importServices(array $services): void
    {
        $bridge = new BridgeContent();

        foreach ($services as $service) {
            $infos = [
                'type' => 'node',
                'bundle' => 'services',
                'title' => trim($service["service"]),
                'structure' => $this->getReference($service, $bridge, 'structure', 'site'),
                'specialties' => $this->getReference($service, $bridge, 'specialties', 'spécialité'),
                'head_service' => $this->getUserReferences($service, 'chef de service'),
                'directions' => trim($service["pôle/direction"]),
                'secretariats' => trim($service["accueil / secrétariat"]),
                'paramedical_contact' => $service['coordo paramédical'],
                'executives' => $this->getUserReferences($service, 'cadre'),
                'referral_ide' => $this->getUserReferences($service, 'ide référente'),
                'administrative_assistants' => trim($service['auxiliaire administrative']),
                'email' => trim($service['email']),
                'care_stations' => trim($service['poste de soins']),
                'nurses' => trim($service["infirmiers"]),
                'internals' => trim($service["internes"]),
                'doctor_calls' => trim($service["médecin de garde"]),
                'night_executives' => trim($service["cadre de nuit"]),
                'medical_advices' => trim($service["avis médical"]),
                'hospitalizations' => trim($service["hospitalisation / assistante médicale"]),
                'consultations' => trim($service["consultations"]),
                'programmings' => trim($service["programmation"]),
                'fax' => trim($service["fax"]),
                'locations' => trim($service["emplacement"])
            ];
            $content = Services::query()->where('title', trim($service['service']))->first();
            if ($content) {
                $bridge->contentUpdate('services', array_merge($infos, [
                    'uuid' => $content->uuid,
                ]));
            } else {
                $bridge->contentInsert('services', $infos);
            }
        }
    }

    /**
     * import services
     */
    public function handle(): void
    {
        /** @var string|null $path */
        $path = $this->option('path');

        if ($path) {
            $this->filePath = $path;
        }

        if (!file_exists($this->filePath)) {
            throw new \Exception('file not found');
        }
        $csv = Reader::createFromPath($this->filePath, 'r');

        $encoding = mb_detect_encoding($csv->getContent(), mb_list_encodings(), true);

        if ($encoding !== 'UTF-8') {
            $csv->setOutputBOM(Reader::BOM_UTF8);
            $csv->addStreamFilter('convert.iconv.'.$encoding.'/UTF-8');
        }

        $csv->setDelimiter(';');
        $csv->setHeaderOffset(0);

        $headers = array_map(
            function ($item) {
                return trim(strtolower($item));
            },
            $csv->getHeader()
        );

        $services = iterator_to_array($csv->getRecords($headers));
        $this->importServices($services);
    }
}
