<?php

use Illuminate\Support\Str;
use Inside\Content\Models\Contents\Structure;
use Inside\Host\Bridge\BridgeContent;

/**
 * @param mixed $user
 * @param BridgeContent $bridge
 * @param string $attribute
 * @param string $contentType
 * @return string|null
 * @throws Exception
 */
function getEntityReference($user, BridgeContent $bridge, string $attribute, string $contentType): ?string
{
    $uuid = null;
    if (!$user->{$attribute}) {
        return null;
    }


    $title = $user->{$attribute};

    $entity = call_user_func(type_to_class($contentType).'::query')
        ->where('title', $title)->first();

    if ($entity) {
        $uuid = $entity->uuid;
    } else {
        $uuid = $bridge->contentInsert($contentType, [
            'type' => 'node',
            'bundle' => $contentType,
            'title' => $title,
        ]);
    }

    return $uuid;
}

return [
    'discovering_attribute' => 'id',
    'postfilter' => function ($users) {
        return $users
            ->reject(function ($user) {
                if (! $user->accountEnabled) {
                    return true;
                }

                $state = Str::lower($user->state);
                $allowedStates = ["paris cadre_cmt", "cmt", "paris cmt", "paris", "hauts-de-seine", "ghpsj"];
                // Generic accounts need to be imported, no matter the filters
                if ($state === "gen") {
                    return false;
                }

                // regular users needs state + mail + surname and firstname
                return !in_array($state, $allowedStates)
                    || !Str::contains($user->mail, "@ghpsj.fr")
                    || empty($user->surname)
                    || empty($user->givenName);
            });
    },
    'sync_attributes' => [
        'name' => function ($user, BridgeContent $bridge) {
            return Str::lower($user->state) === "gen" ? $user->userPrincipalName : $user->mail;
        },
        'mail' => function ($user, BridgeContent $bridge) {
            return Str::lower($user->state) === "gen" ? $user->userPrincipalName : $user->mail;
        },
        'api_id' => 'id',
        'langcode' => function ($user, BridgeContent $bridge) {
            return 'fr';
        },
        'firstname' => function ($user, BridgeContent $bridge) {
            if (!isset($user->givenName)) {
                return null;
            }

            return ucwords(Str::lower($user->givenName), "- ");
        },
        'lastname' => 'surname',
        'generic_account' => function ($user, BridgeContent $bridge) {
            return Str::lower($user->state) === "gen";
        },
        'departments' => function ($user, BridgeContent $bridge) {
            return getEntityReference($user, $bridge, 'department', 'departments');
        },
        'structure' => function ($user, BridgeContent $bridge) {
            $state = Str::lower($user->state);

            switch ($state) {
                case 'paris':
                    $site = "Hôpital Paris Saint-Joseph";
                    break;
                case 'hauts-de-seine':
                    $site = "Hôpital Marie-Lannelongue";
                    break;
                case 'ghpsj':
                    $site = "Hôpitaux Saint-Joseph Marie-Lannelongue";
                    break;
                case 'paris cadre_cmt':
                case 'cmt':
                case 'paris cmt':
                    $site = "Centres de Santé Marie-Thérèse";
                    break;
                default:
                    $site = null;
                    break;
            }

            if (!$site) {
                return null;
            }

            $structure = Structure::query()->where('title', $site)->first();
            if ($structure) {
                $uuid = $structure->uuid;
            } else {
                $uuid = $bridge->contentInsert('structure', [
                    'type' => 'node',
                    'bundle' => 'structure',
                    'title' => $site,
                ]);
            }
            return $uuid;
        },
        'upn' => 'userPrincipalName',
        'samaccountname' => 'onPremisesSamAccountName',
    ],
];
