<?php

namespace Inside\HARE\Services;

use Barryvdh\DomPDF\Facade as PDF;
use Illuminate\Support\Facades\DB;
use Illuminate\Support\Facades\Storage;
use Illuminate\Support\Facades\View;
use Inside\Authentication\Models\User;
use Inside\Content\Models\Contents\ExperienceCategories;
use Inside\Content\Models\Contents\Experiences;
use Inside\Content\Models\Contents\ExperienceYear;

/**
 * PDF service.
 *
 * @category Class
 * @package  Inside\HARE\Services\PdfService
 * @author   Maecia <technique@maecia.com>
 * @license  http://www.gnu.org/copyleft/gpl.html GNU General Public License
 * @link     http://www.maecia.com/
 */
class PdfService
{
    protected function cleanInfos(array $pdfInfos): array
    {
        foreach ($pdfInfos as $key => $field) {
            if (!is_array($pdfInfos[$key]) && strpos($pdfInfos[$key], "<meta") !== false) {
                $pdfInfos[$key] = strstr($pdfInfos[$key], '>') ?: '';
                if ($pdfInfos[$key][1] === "<") {
                    $pdfInfos[$key] = ltrim($pdfInfos[$key], $pdfInfos[$key][0]);
                } else {
                    $pdfInfos[$key] = substr_replace($pdfInfos[$key], '<p>', 0, 1);
                    $pdfInfos[$key] .= "</p>";
                }
            }
        }
        return $pdfInfos;
    }

    /**
     * @param mixed $experience
     * @param bool $update
     * @return void
     */
    public function generatePdf($experience, bool $update = false)
    {
        if (empty($experience)) {
            return;
        }

        $pdfInfos = $experience->getAttributes();
        $pdfInfos['year'] = null;
        $pdfInfos['referents'] = [];
        $pdfInfos['categories'] = [];
        $related_year = DB::table('inside_pivots')
            ->where('related_type', 'Inside\Content\Models\Contents\ExperienceYear')
            ->where('related_field', 'year')
            ->where('parent_uuid', $experience->uuid)->pluck('related_uuid')->first();
        $related_referents = DB::table('inside_pivots')
            ->where('related_field', 'referents')
            ->where('related_type', 'Inside\Content\Models\Contents\Users')
            ->where('parent_uuid', $experience->uuid)->pluck('related_uuid')->toArray();
        $related_categories = DB::table('inside_pivots')
            ->where('related_field', 'experience_categories')
            ->where('related_type', 'Inside\Content\Models\Contents\ExperienceCategories')
            ->where('parent_uuid', $experience->uuid)->pluck('related_uuid')->toArray();

        $experience_year_model = ExperienceYear::find($related_year);

        if ($experience_year_model) {
            $experience_year = $experience_year_model->getAttributes();
            $pdfInfos["year"] = $experience_year['title'];
        }
        if ($related_referents) {
            $experience_referents = [];
            foreach ($related_referents as $referent) {
                $user = User::find($referent);
                if ($user instanceof User) {
                    $experience_referents[] = $user->name;
                }
            }
            $pdfInfos["referents"] = $experience_referents;
        }

        if ($related_categories) {
            $categories = [];
            foreach ($related_categories as $related_category) {
                $category = ExperienceCategories::find($related_category);
                if ($category) {
                    array_push($categories, $category->title);
                }
            }
            $pdfInfos["categories"] = $categories;
        }
        $pdfInfos = $this->cleanInfos($pdfInfos);

        $filename = "experience-" . $experience->uuid . ".pdf";
        if (!$update && Storage::disk('protected')->exists('experiences/' . $filename)) {
            return;
        }
        $view = View::make('experience', $pdfInfos);
        $html = $view->render();
        $html = preg_replace('/>\s+</', '><', $html);

        $pdf = PDF::loadHTML($html);
        Storage::disk('protected')->put('experiences/' . $filename, $pdf->download()->getOriginalContent());
        $pdf = null;
        unset($pdf);
    }
}
