<?php

namespace Inside\HARE\Http\Middleware;

use Illuminate\Http\Request;
use GuzzleHttp\Client as HttpClient;
use Illuminate\Support\Facades\Cache;
use Closure;

/**
 * Weather middleware adds weather data on the weather_cities models.
 *
 * @category Class
 * @license  http://www.gnu.org/copyleft/gpl.html GNU General Public License
 * @link     http://www.maecia.com/
 */
class WeatherData
{
    /**
     * Api key for https://openweathermap.org
     * @var string
     */
    protected $api_key = 'da9f01fad3055b91474533688676c915';

    /**
     * Icon mapping for understandable names.
     * https://openweathermap.org/weather-conditions
     * https://codepen.io/noahblon/pen/lxukH
     * @var array
     */
    protected $icon_mapping = [
        '01d' => 'sun',
        '01n' => 'moon',
        '02d' => 'cloudSun',
        '02n' => 'cloudMoon',
        '03d' => 'cloud',
        '03n' => 'cloud',
        '04d' => 'cloud',
        '04n' => 'cloud',
        '09d' => 'cloudDrizzle',
        '09n' => 'cloudDrizzle',
        '10d' => 'cloudDrizzleSun',
        '10n' => 'cloudDrizzleMoon',
        '11d' => 'cloudLightning',
        '11n' => 'cloudLightning',
        '13d' => 'cloudSnowAlt',
        '13n' => 'cloudSnowAlt',
        '50d' => 'cloudFog',
        '50n' => 'cloudFog',
    ];

    /**
     * Handle an incoming request.
     *
     * @param Request $request
     * @param  \Closure  $next
     * @return mixed
     */
    public function handle(Request $request, Closure $next)
    {
        $response = $next($request);

        if ($request->method() === 'GET' && $request->path() === 'api/v1/content/weather_cities') {
            $content = json_decode_response($response);
            $this->addWeatherInfo($content['data']);
            set_response($response, $content);
        }

        return $response;
    }

    /**
     * Add 'temp' and 'icon' on the cities
     * @param array $cities
     */
    protected function addWeatherInfo(array &$cities): void
    {
        // Fetch weather apis
        $client = new HttpClient();
        $promises = array_map(function ($city) use ($client) {
            if (empty($city['title'])) {
                return;
            }

            // Build request url
            $url = 'https://api.openweathermap.org/data/2.5/weather?units=metric&q=';
            $url .= urlencode($city['title']);
            if (!empty($city['country'])) {
                $url .= ','.urlencode($city['country']);
            }
            $url .= '&APPID='.$this->api_key;

            // Cache
            if ($cached = Cache::tags(['weather'])->get($url)) {
                return new \GuzzleHttp\Promise\FulfilledPromise($cached);
            }

            // Call api, decode body, save in cache
            return $client->sendAsync(new \GuzzleHttp\Psr7\Request('GET', $url))->then(function ($res) {
                return json_decode($res->getBody(), true);
            }, function () {
                return [];
            })->then(function ($res) use ($url) {
                Cache::tags(['weather'])->put($url, $res, 30);
                return $res;
            });
        }, $cities);

        // Wait till all the requests are finished
        $results = \GuzzleHttp\Promise\all($promises)->wait();

        // Fill cities data
        foreach ($cities as $i => &$city) {
            $data = $results[$i];

            // Temp
            $city['temp'] = isset($data['main']['temp']) ? round($data['main']['temp']) : '--';

            // Icon
            $city['icon'] = 'unkown';
            if (isset($data['weather'][0]['icon'])) {
                $icon = $data['weather'][0]['icon'];
                if (isset($this->icon_mapping[$icon])) {
                    $city['icon'] = $this->icon_mapping[$icon];
                }
            }
        }
    }
}
