<?php

use Adldap\Models\Entry;
use Inside\Content\Models\Contents\Direction;
use Inside\Host\Bridge\BridgeContent;
use Inside\Content\Models\Contents\Users;
use Illuminate\Support\Carbon;
use Illuminate\Support\Facades\File;
use Illuminate\Support\Str;

function get_ldap_reference(Entry $user, BridgeContent $bridge, string $attribute, string $entityName): ?array
{
    if (!$user->hasAttribute($attribute)) {
        return null;
    }

    $references = [];

    $userReferences = $user->getAttribute($attribute);

    foreach ($userReferences as $referenceName) {
        if (empty($referenceName)) {
            continue;
        }

        $uuid      = false;
        $query     = call_user_func(type_to_class($entityName) . '::query');
        $reference = $query->where('title', $referenceName)->first();

        if ($reference) {
            $uuid = $reference->uuid;
        } else {
            $uuidBridge = $bridge->contentInsert(
                $entityName,
                [
                    'type'   => 'node',
                    'bundle' => $entityName,
                    'title'  => $referenceName,
                    'uid'    => 1,
                ]
            );

            $reference = call_user_func(type_to_class($entityName) . '::find', $uuidBridge);

            if ($reference) {
                $uuid = $reference->uuid;
            }
        }

        if ($uuid) {
            $references[] = $uuid;
        }
    }

    if (!empty($references)) {
        return $references;
    }

    return null;
}

function get_direction_references(Entry $user, BridgeContent $bridge): ?array
{
    if (!$user->hasAttribute('description')) {
        return null;
    }
    $references = [];

    $userReferences = $user->getAttribute('description');

    foreach ($userReferences as $referenceName) {
        if (empty($referenceName)) {
            continue;
        }

        $uuid      = false;
        $reference = Direction::where('title', $referenceName)->first();

        if ($reference) {
            $references[] = $reference->uuid;
            if ($reference->direction && $reference->direction->isNotEmpty()) {
                $references[] = $reference->direction->first()->uuid;
            }
        }
    }
    if (!empty($references)) {
        return $references;
    }

    return null;
}

return [
    'connection'      => env('ADLDAP_CONNECTION', 'default'),
    'provider'        => Adldap\Laravel\Auth\DatabaseUserProvider::class,
    'rules'           => [
        Adldap\Laravel\Validation\Rules\DenyTrashed::class,
    ],
    'scopes'          => [],
    'usernames'       => [
        'ldap'     => [
            'discover'     => env('ADLDAP_LDAP_DISCOVER', 'mail'),
            'authenticate' => env('ADLDAP_LDAP_AUTHENTICATE', 'mail'),
        ],
        'eloquent' => env('ADLDAP_ELOQUENT_VALUE', 'email'),
        'windows'  => [
            'discover' => env('ADLDAP_WINDOWS_DISCOVER', 'samaccountname'),
            'key'      => env('ADLDAP_WINDOWS_KEY', 'AUTH_USER'),
        ],
    ],
    'passwords'       => [
        'sync'   => env('ADLDAP_PASSWORD_SYNC', false),
        'column' => 'password',
    ],
    'login_fallback'  => env('ADLDAP_LOGIN_FALLBACK', false),
    'sync_attributes' => [
        'mail'        => 'mail',
        'name'        => 'samaccountname',
        'firstname'   => 'givenname',
        'lastname'    => 'sn',
        'function'    => 'title',
        'phone'       => function (Entry $user, BridgeContent $bridge) {
            $phone = $user->getFirstAttribute('telephonenumber');

            return $phone && substr($phone, 0, 3) == "+33" ? substr_replace($phone, "0", 0, 3) : $phone;
        },
        'birthday'    => function (Entry $user, BridgeContent $bridge) {
            if (!$user->hasAttribute('samaccountname')) {
                return null;
            }
            $samaccountname = $user->getFirstAttribute('samaccountname');
            if ($samaccountname) {
                $user = Users::where('name', $samaccountname)->first();
                if ($user && $user->birthday != null) {
                    return Carbon::createFromTimestamp($user->birthday)->format('Y-m-d H:i:s');
                }
            }

            return null;
        },
        //'mobile'      => 'mobile',
        'mobile'      => function (Entry $user, BridgeContent $bridge) {
            $mobile = $user->getFirstAttribute('mobile');

            return $mobile && substr($mobile, 0, 3) == "+33" ? substr_replace($mobile, "0", 0, 3) : $mobile;
        },
        'services'    => function (Entry $user, BridgeContent $bridge) {
            return get_ldap_reference($user, $bridge, 'department', 'services');
        },
        'direction'   => function (Entry $user, BridgeContent $bridge) {
            return get_direction_references($user, $bridge);
        },
        'workplace'   => function (Entry $user, BridgeContent $bridge) {
            return get_ldap_reference($user, $bridge, 'l', 'workplace');
        },
        'emplacement' => 'office',
        'image' => function (Entry $user, BridgeContent $bridge) {
            $insideUser = Users::query()->where('email', $user->getFirstAttribute('mail'))->first();
            $imageName = Str::lower(trim($user->getFirstAttribute('sn'))) . '_' . Str::lower(trim($user->getFirstAttribute('givenname'))) . '.jpg';

            /**
             * If the user already has his picture set, return false.
             * The image attribute will then be removed in the post_sync callback
             */
            if ($insideUser) {
                $userImage = $insideUser->getAttribute('image');
                if (!empty($userImage) && Str::contains($userImage, $imageName)) {
                    return false;
                }
            }

            $imgPath = cms_base_path(env('APP_STORAGE_PATH', storage_path()) . '/profile/' . strtolower($imageName));
            $chunkId    = Str::random(32);
            $chunksPath = cms_base_path(env('APP_STORAGE_PATH', storage_path()) . '/chunks/' . $chunkId);

            File::makeDirectory($chunksPath);
            if (!File::isFile($imgPath) || !File::exists($imgPath)) {
                return false;
            }

            File::copy($imgPath, $chunksPath . '/' . $imageName);
            return 'chunks/' . $chunkId . '/' . $imageName;
        }
    ],
    'cleandata' => function (&$data) {
        if ($data['image'] === false) {
            unset($data['image']);
        }
    },
    'logging'         => [
        'enabled' => true,
        'events'  => [
            \Adldap\Laravel\Events\Importing::class                 => \Adldap\Laravel\Listeners\LogImport::class,
            \Adldap\Laravel\Events\Synchronized::class              => \Adldap\Laravel\Listeners\LogSynchronized::class,
            \Adldap\Laravel\Events\Synchronizing::class             => \Adldap\Laravel\Listeners\LogSynchronizing::class,
            \Adldap\Laravel\Events\Authenticated::class             => \Adldap\Laravel\Listeners\LogAuthenticated::class,
            \Adldap\Laravel\Events\Authenticating::class            => \Adldap\Laravel\Listeners\LogAuthentication::class,
            \Adldap\Laravel\Events\AuthenticationFailed::class      => \Adldap\Laravel\Listeners\LogAuthenticationFailure::class,
            \Adldap\Laravel\Events\AuthenticationRejected::class    => \Adldap\Laravel\Listeners\LogAuthenticationRejection::class,
            \Adldap\Laravel\Events\AuthenticationSuccessful::class  => \Adldap\Laravel\Listeners\LogAuthenticationSuccess::class,
            \Adldap\Laravel\Events\DiscoveredWithCredentials::class => \Adldap\Laravel\Listeners\LogDiscovery::class,
            \Adldap\Laravel\Events\AuthenticatedWithWindows::class  => \Adldap\Laravel\Listeners\LogWindowsAuth::class,
            \Adldap\Laravel\Events\AuthenticatedModelTrashed::class => \Adldap\Laravel\Listeners\LogTrashedModel::class,
        ],
    ],
];

/**
UPDATE `inside_users` SET provider_type = 'ldap', provider_name = 'default'
WHERE `provider_type` IS NULL AND `email` LIKE '%@sia-habitat.com' OR `email` LIKE '%@escaut-habitat.com' OR  `email` LIKE '%@nomail.sia-habitat.com'

 */
