<?php

use Adldap\Models\Entry;
use Inside\Host\Bridge\BridgeContent;

if (!function_exists('get_ldap_reference')) {
    function get_ldap_reference(Entry $user, BridgeContent $bridge, string $attribute, string $contentType): ?array
    {
        if (!$user->hasAttribute($attribute)) {
            return null;
        }

        $uuids = [];
        $titles = $user->getAttribute($attribute);

        foreach ($titles as $title) {
            $entity = call_user_func(type_to_class($contentType).'::query')
                ->where('title', $title)
                ->first();

            if ($entity) {
                $uuid = $entity->uuid;
            } else {
                $uuid = $bridge->contentInsert(
                    $contentType,
                    [
                        'type' => 'node',
                        'bundle' => $contentType,
                        'title' => $title,
                        'uid' => 1,
                    ]
                );
            }

            $uuids[] = $uuid;
        }

        return empty($uuids) ? null : $uuids;
    }
}

return [
    'usernames' => [
        'ldap' => [
            'discover' => env('ADLDAP_LDAP_DISCOVER', 'mail'),
            'authenticate' => env('ADLDAP_LDAP_AUTHENTICATE', 'mail'),
        ],
        'eloquent' => env('ADLDAP_ELOQUENT_VALUE', 'email'),
    ],
    'discovering_attributes' => 'name',
    'sync_attributes' => [
        'mail' => 'mail',
        'is_maintenance' => function () {
            return false;
        },
        'name' => 'userprincipalname',
        'firstname' => 'givenname',
        'lastname' => 'sn',
        'fullname' => 'cn',
        'function' => 'title',
        'phone' => function (Entry $user) {
            return $user['telephonenumber'][0];
        },
        'mobile_phone' => function (Entry $user) {
            return $user['mobile'][0];
        },
        'locations' => function (Entry $user, BridgeContent $bridge) {
            return get_ldap_reference($user, $bridge, 'physicaldeliveryofficename', 'locations');
        },
        'entities' => function (Entry $user, BridgeContent $bridge) {
            return get_ldap_reference($user, $bridge, 'company', 'entities');
        },
    ],
    'logging' => [
        'enabled' => true,
        'events' => [
            \Adldap\Laravel\Events\Importing::class => \Adldap\Laravel\Listeners\LogImport::class,
            \Adldap\Laravel\Events\Synchronized::class => \Adldap\Laravel\Listeners\LogSynchronized::class,
            \Adldap\Laravel\Events\Synchronizing::class => \Adldap\Laravel\Listeners\LogSynchronizing::class,
            \Adldap\Laravel\Events\Authenticated::class => \Adldap\Laravel\Listeners\LogAuthenticated::class,
            \Adldap\Laravel\Events\Authenticating::class => \Adldap\Laravel\Listeners\LogAuthentication::class,
            \Adldap\Laravel\Events\AuthenticationFailed::class => \Adldap\Laravel\Listeners\LogAuthenticationFailure::class,
            \Adldap\Laravel\Events\AuthenticationRejected::class => \Adldap\Laravel\Listeners\LogAuthenticationRejection::class,
            \Adldap\Laravel\Events\AuthenticationSuccessful::class => \Adldap\Laravel\Listeners\LogAuthenticationSuccess::class,
            \Adldap\Laravel\Events\DiscoveredWithCredentials::class => \Adldap\Laravel\Listeners\LogDiscovery::class,
            \Adldap\Laravel\Events\AuthenticatedWithWindows::class => \Adldap\Laravel\Listeners\LogWindowsAuth::class,
            \Adldap\Laravel\Events\AuthenticatedModelTrashed::class => \Adldap\Laravel\Listeners\LogTrashedModel::class,
        ],
    ],
];
