<?php

declare(strict_types=1);

use Illuminate\Database\Migrations\Migration;
use Illuminate\Support\Facades\Artisan;
use Inside\Content\Models\Contents\Tools;
use Inside\Content\Models\Contents\Users;
use Inside\Host\Bridge\BridgeContent;
use Inside\Host\Migrations\CreateContentTypes;

final class SeedDefaulttools extends Migration
{
    use CreateContentTypes;

    public function up(): void
    {
        $this->bootUp(__FILE__);
        Artisan::call('models:rebuild');

        $tools = [
            ['fr' => 'Outil 1', 'nl' => 'Tool 1'],
            ['fr' => 'Outil 2', 'nl' => 'Tool 2'],
            ['fr' => 'Outil 3', 'nl' => 'Tool 3'],
            ['fr' => 'Outil 4', 'nl' => 'Tool 4'],
            ['fr' => 'Outil 5', 'nl' => 'Tool 5'],
        ];

        $bridgeContent = new BridgeContent();
        $tools = collect($tools)->map(fn (array $titles) => $this->getToolUuids($bridgeContent, $titles));

        $this->getBridge()?->contentTypeCreateOrUpdateField('users', 'tools', [
            'default' => $tools->pluck('uuid_host')->map(fn (string $uuid_host) => ['target_uuid' => $uuid_host])->all(),
        ]);

        Users::each(function (Users $user) use ($bridgeContent, $tools) {
            $defaultTools = $tools->pluck($user->langcode);

            $bridgeContent->contentUpdate('users', [
                'uuid' => $user->uuid,
                'tools' => $user->tools->pluck('uuid')->merge($defaultTools)->filter()->unique()->values()->all(),
            ]);
        });

        $this->finishUp();
    }

    private function getToolUuids(BridgeContent $bridge, array $titles): array
    {
        $locale = config('app.locale');
        $languages = collect(list_languages())->diff([$locale]);

        $original = Tools::where(['title' => $titles[$locale], 'langcode' => $locale])->first();

        if (!$original instanceof Tools) {
            $original = $bridge->contentInsert('tools', ['title' => $titles[$locale], 'langcode' => $locale]);
            $original = Tools::findOrFail($original);
        }

        // collection of type [langcode => uuid]
        $uuids = $languages->mapWithKeys(function (string $langcode) use ($bridge, $titles, $original) {
            $data = [
                'title' => $titles[$langcode],
                'langcode' => $langcode,
                'uuid_host' => $original->uuid_host,
            ];

            $uuid = Tools::where($data)->first()?->uuid ?? $bridge->contentInsert('tools', $data);
            return [$langcode => $uuid];
        });

        return $uuids->merge([$locale => $original->uuid, 'uuid_host' => $original->uuid_host])->all();
    }
}
