<?php

declare(strict_types=1);

use Illuminate\Database\Migrations\Migration;
use Illuminate\Support\Carbon;
use Illuminate\Support\Facades\Artisan;
use Inside\Content\Facades\Schema;
use Inside\Content\Models\Contents\Comments;
use Inside\Content\Models\Contents\InternalNews;
use Inside\Content\Models\Section;
use Inside\Host\Bridge\BridgeContent;
use Inside\Host\Bridge\BridgeContentType;
use Inside\Host\Migrations\CreateContentTypes;
use Inside\Reaction\Models\Reaction;
use Inside\Support\Str;

final class RemoveCominternsManageNewsCategories extends Migration
{
    use CreateContentTypes;

    private function transformSection(Section $section): array
    {
        $transformed = $section->toArray();
        $transformed['bundle'] = $section->section_type;
        $transformed['pgID'] = '_'.Str::random(8);

        if ($section->section_content?->isNotEmpty()) {
            $transformed['content'] = $section->section_content->map(fn (Section $section) => $this->transformSection($section))->all();
        }

        return $transformed;
    }

    private function copyInternalNewsToNews(): void
    {
        $bridgeContent = new BridgeContent();

        InternalNews::get()->each(function (InternalNews $internalNews) use ($bridgeContent) {
            // Copy Content
            $uuid = $bridgeContent->contentInsert('news', [
                'title' => $internalNews->title,
                'comments' => $internalNews->comments,
                'content' => $internalNews->section_content->map(fn (Section $section) => $this->transformSection($section))->all(),
                'image' => $internalNews->image,
                'chapo' => $internalNews->chapo,
                'published_at' => Carbon::createFromTimestamp($internalNews->published_at)->toDateTimeString(),
                'created_at' => Carbon::createFromTimestamp($internalNews->created_at)->toDateTimeString(),
                'author' => $internalNews->author_id,
                'status' => $internalNews->status,
            ]);

            // Copy Comments
            $newComments = [];
            $internalNews->comments
                ->sortBy('pid')
                ->each(function (Comments $comment) use ($bridgeContent, $uuid, &$newComments) {
                    $newComment = $bridgeContent->contentInsert($comment->content_type, [
                        'bundle' => 'news',
                        'news' => [$uuid],
                        'body' => $comment->body,
                        'file' => $comment->file,
                        'langcode' => $comment->langcode,
                        'status' => $comment->status,
                        'author' => $comment->author_id,
                        'created_at' => Carbon::createFromTimestamp($comment->created_at)->toDateTimeString(),
                        'pid' => $newComments[$comment->pid] ?? null,
                    ]);

                    $newComments[$comment->uuid] = $newComment;
                });

            // Copy Reactions
            Reaction::query()
                ->where('reactionable_uuid', $internalNews->uuid)
                ->where('reactionable_type', get_class($internalNews))
                ->each(fn (Reaction $reaction) => $reaction->update(['reactionable_uuid' => $uuid, 'reactionable_type' => type_to_class('news')]));
        });
    }

    public function up(): void
    {
        $this->bootUp(__FILE__);
        $this->migrateContent('news_categories');
        $this->migrateContent('collab_corner');

        $this->getBridge()?->contentTypeCreateOrUpdateField(
            'news',
            'news_categories',
            [
                'name' => 'news_categories',
                'searchable' => false,
                'translatable' => true,
                'search_result_field' => true,
                'search_filter' => true,
                'required' => false,
                'weight' => 6,
                'title' => [
                    'fr' => 'Catégories',
                ],
                'widget' => [
                    'type' => BridgeContentType::SELECT_WIDGET,
                    'settings' => [
                        'size' => 60,
                        'placeholder' => '',
                    ],
                    'weight' => 6,
                ],
                'type' => BridgeContentType::REFERENCE_FIELD,
                'settings' => [
                    'target_type' => ['news_categories'],
                    'cardinality' => 1,
                ],
                'selectable_all' => false,
            ]
        );

        Artisan::call('models:rebuild');

        if (Schema::hasModel('internal_news')) {
            $this->copyInternalNewsToNews();
            $this->getBridge()?->contentTypeDelete('internal_news');
        }

        $this->finishUp();
    }
}
