<?php

use Illuminate\Database\Migrations\Migration;
use Illuminate\Support\Facades\Artisan;
use Inside\Content\Facades\Schema as InsideSchema;
use Inside\Content\Models\Contents\Custom;
use Inside\Content\Models\Contents\ToolsCategories;
use Inside\Host\Bridge\BridgeContent;
use Inside\Host\Bridge\BridgeContentType;
use Inside\Host\Migrations\CreateContentTypes;
use Inside\Menu\Models\HeaderMenuLink;
use Inside\Slug\Models\Slug;

class FixListingToolNotCreated extends Migration
{
    use CreateContentTypes;

    public function up(): void
    {
        $this->bootUp(__FILE__);

        if (InsideSchema::hasContentType('tools_categories_listings')) {
            if (DB::table('inside_content_tools_categories_listings')->exists()) {
                return;
            }
        }

        if (! InsideSchema::hasContentType('tools') || ! InsideSchema::hasContentType('tools_categories')) {
            return;
        }

        $bridge = new BridgeContentType();
        $bridgeContent = new BridgeContent();

        $this->migrateContent('tools_categories_listings');
        Artisan::call('models:rebuild');

        $bridge->contentTypeCreateOrUpdateField(
            type: 'tools_categories',
            fieldName: 'tools_categories_listings',
            options:[
                'name' => 'tools_categories_listings',
                'searchable' => true,
                'translatable' => true,
                'search_result_field' => false,
                'searchable_filter' => false,
                'required' => true,
                'weight' => -2,
                'title' => [
                    'fr' => 'Listing catégorie d\'outils',
                    'en' => 'Tools categories listing',
                    'es' => 'Listado de categorías de herramientas',
                    'pt' => 'Listagem de categorias de ferramentas',
                    'de' => 'Werkzeugkategorienliste',
                    'nl' => 'Lijst van gereedschapscategorieën',
                ],
                'widget' => [
                    'type' => BridgeContentType::SELECT_WIDGET,
                    'settings' => [],
                    'weight' => 1,
                ],
                'type' => BridgeContentType::REFERENCE_FIELD,
                'settings' => [
                    'target_type' => ['tools_categories_listings'],
                    'cardinality' => 1,
                ],
            ],
            weight: -2
        );

        $toolsLang = [
            'fr' => 'Outils',
            'en' => 'Tools',
            'es' => 'Herramientas',
            'pt' => 'Ferramentas',
            'de' => 'Werkzeuge',
            'nl' => 'Gereedschap',
        ];

        $langs = array_unique(array_merge([config('app.locale', 'fr')], list_languages()));

        foreach ($langs as $lang) {
            Slug::where('slug', strtolower($toolsLang[$lang]))
                ->update(['slug' => strtolower($toolsLang[$lang]).'_old']);

            $uuidHost = DB::table('inside_content_tools_categories_listings')
                ->where('langcode', config('app.locale', 'fr'))
                ->first()
                ->uuid_host;

            $data = [
                'title' => $toolsLang[$lang],
                'langcode' => $lang,
            ];

            if ($uuidHost) {
                $data['uuid_host'] = $uuidHost;
            }

            $toolsCategoriesListings = $bridgeContent->contentInsert(
                type: 'tools_categories_listings',
                data: $data
            );

            $toolsCategories = ToolsCategories::where('langcode', $lang)
                ->pluck('uuid', 'title')
                ->each(function ($uuid, $title) use ($bridgeContent, $toolsCategoriesListings) {
                    $bridgeContent->contentInsert(
                        type: 'tools_categories',
                        data: [
                            'title' => $title,
                            'uuid' => $uuid,
                            'tools_categories_listings' => $toolsCategoriesListings,
                        ]
                    );
                });

            $menuHeader = HeaderMenuLink::query()->where('page', 'ToolsPage')
                ->where('langcode', $lang)
                ->get();

            if ($menuHeader->isNotEmpty()) {
                $menuHeader->each(fn ($menu) => $menu->update([
                    'link' => 'tools_categories_listings|'.$toolsCategoriesListings,
                    'page' => null,
                ]));
            }
        }

        Artisan::call('models:rebuild');
        $this->finishUp();
    }
}
