<?php

namespace Inside\ECEN\Listeners;

use Illuminate\Contracts\Queue\ShouldQueue;
use Illuminate\Support\Str;
use Inside\Content\Events\ContentFullyInsertedEvent;
use Inside\Content\Facades\Schema;
use Inside\Host\Bridge\BridgeContent;

class ContentCreatedListener extends NotificationGedListener implements ShouldQueue
{
    /**
     * @param ContentFullyInsertedEvent $event
     * @throws \Exception
     */
    public function handle(ContentFullyInsertedEvent $event): void
    {
        if (!in_array($event->model->content_type, ['sites', 'clients', 'shared_folders', 'shared_documents', 'space_folders', 'space_documents'])) {
            return;
        }

        $model = $event->model;
        $langcode = $model->langcode === 'en' ? 'fr' : 'en';

        if (in_array(class_to_type($model), ['shared_documents', 'space_documents'])) {
            $this->notification($model, 'createDocuments', true);
        }

        $translation = $model->getTranslationIfExists($langcode);

        if ($translation->langcode !== $model->langcode) {
            return;
        }

        $fields = [
            'color' => 'text',
            'sites' => 'reference',
            'clients' => 'reference',
            'content' => 'sections',
            'file' => 'file',
            'shared_folders' => 'reference',
            'space_folders' => 'reference',
            'image' => 'file',
        ];

        $data = [
            'type' => 'node',
            'bundle' => class_to_type($model),
            'uuid_host' => $model->uuid_host,
            'langcode' => $langcode,
            'title' => $model->title,
            'status' => $model->status,
            'author' => $model->author,
        ];

        if ($model->content_type === 'clients') {
            $data['machine_name'] = Str::slug(trim($model->title));
        }

        foreach ($fields as $field => $type) {
            if (!Schema::hasField($data['bundle'], $field)) {
                continue;
            }

            switch ($type) {
                case 'file':
                case 'text':
                    $data[$field] = $model->{$field};
                    break;
                case 'sections':
                    $sourceSections = $model->{Str::camel('section_' . $field)} ?? [];

                    $content = [];
                    foreach ($sourceSections as $sourceSection) {
                        $section = [
                            'bundle' => str_replace('inside_section_', '', $sourceSection->getTable()),
                        ];

                        $section = array_merge($section, $sourceSection->toArray());

                        $sourceSection = call_user_func(get_class($sourceSection) . '::find', $sourceSection->uuid);

                        if (isset($sourceSection->sectionContent)) {
                            $section['content'] = [];

                            $subSections = $sourceSection->sectionContent;

                            foreach ($subSections as $subSectionContent) {
                                $subSection = [
                                    'bundle' => str_replace('inside_section_', '', $subSectionContent->getTable()),
                                ];

                                $subSection = array_merge($subSection, $subSectionContent->toArray());

                                $section['content'][] = $subSection;

                                $subSectionContent = null;
                            }

                            $subSections = null;
                        }

                        $sourceSection = null;

                        $content[] = $section;
                    }

                    $sourceSections = null;

                    if (!empty($content)) {
                        $data[$field] = $content;
                        $content = null;
                        $section = null;
                        $subSection = null;
                    }

                    break;
                case 'reference':
                    if (!Schema::hasField($data['bundle'], $field)) {
                        continue 2;
                    }

                    $fieldOptions = Schema::getFieldOptions($data['bundle'], $field);
                    $references = $model->{Str::camel($field)};
                    $data[$field] = [];

                    foreach ($references as $reference) {
                        $reference = $reference->getTranslationIfExist($langcode);

                        if ($reference->langcode !== $langcode) {
                            continue;
                        }

                        if ($fieldOptions['cardinality'] === 1) {
                            $data[$field] = $reference->uuid;
                        } else {
                            $data[$field][] = $reference->uuid;
                        }
                    }
                    break;
            }
        }

        $bridge = new BridgeContent();

        $bridge->contentInsert($data['bundle'], $data);
    }
}
