<?php

namespace Inside\ECEN\Http\Middlewares;

use Closure;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\DB;
use Inside\Content\Models\Content;
use Inside\Content\Models\Contents\Users;
use Symfony\Component\HttpFoundation\Response;

/**
 * Custom slugs for BCLH contents
 *
 * @category Class
 * @package  Inside\Content\Http\Middleware\MembersFieldMiddleware
 * @author   Maecia <technique@maecia.com>
 * @license  http://www.gnu.org/copyleft/gpl.html GNU General Public License
 * @link     http://www.maecia.com/
 */
class GedMiddleware
{
    /**
     * @var string[]
     */
    protected $contentTypes = [
        'shared_folders',
        'space_folders',
        'shared_documents',
        'space_documents',
        'dashboards',
        'cenexi_notes',
    ];

    /**
     * Handle an incoming request.
     *
     * @param Request $request
     * @param \Closure $next
     *
     * @return mixed
     */
    public function handle(Request $request, Closure $next)
    {
        $currentPath = $request->path();

        $paths = [
            'api/v1/content',
            'api/v1/search',
        ];

        $applyGedFilters = false;
        $isSearch = false;
        $isSave = false;

        foreach ($paths as $path) {
            if (strpos($currentPath, $path) === 0) {
                switch ($path) {
                    case 'api/v1/search':
                        $isSearch = true;
                        break;
                    case 'api/v1/content':
                        if ($request->getMethod() !== 'GET') {
                            $isSave = true;
                        }
                        break;
                }

                $applyGedFilters = true;
                break;
            }
        }

        if (!$applyGedFilters) {
            return $next($request);
        }

        if (strpos($currentPath, 'api/v1/content/asset') === 0) {
            return $next($request);
        }

        $response = $next($request);
        if ($response instanceof Response && $response->getStatusCode() != 200) {
            return $response;
        }

        /** @var array $content */
        $content = json_decode_response($response);

        if (!$isSave && empty($content['data'])) {
            return $response;
        }

        if ($isSave) {
            $content = ['data' => [$content]];
        }

        if ($isSearch && ($request->has('currentSite') || $request->has('currentClient'))) {
            $parameters = ['currentSite' => 'sites', 'currentClient' => 'clients'];

            foreach ($parameters as $parameter => $field) {
                $currentUuid = $request->has($parameter) ? $request->get($parameter) : false;

                if ($currentUuid) {
                    $searchData = ['search' => $content];
                    $this->handleUuidFilter($searchData, $currentUuid, $field, true);
                    $content = ['data' => array_values($searchData['search']['data'])];
                }
            }
        }

        foreach ($content['data'] as $key => &$data) {
            if ($isSave) {
                $data['content_type'] = str_replace([
                    'api/v1/content/',
                    '/',
                    $data['uuid'],
                ], [
                    '',
                    '',
                    '',
                ], $currentPath);
            }

            if (isset($data['content_type']) && in_array($data['content_type'], ['space_folders', 'shared_folders'])) {
                $parameters = ['currentSite' => 'sites', 'currentClient' => 'clients'];

                foreach ($parameters as $parameter => $field) {
                    $currentUuid = $request->has($parameter) ? $request->get($parameter) : false;

                    if ($currentUuid) {
                        $this->handleUuidFilter($data, $currentUuid, $field);

                        if (array_key_exists('shared_documents', $data)) {
                            $this->handleUuidFilter($data['shared_documents']['data'], $currentUuid, $field);
                        }
                        if (array_key_exists('space_documents', $data)) {
                            $this->handleUuidFilter($data['space_documents']['data'], $currentUuid, $field);
                        }
                    }
                }
            }
        }

        if ($isSave) {
            $content = $content['data']['0'];
        }

        return response()->json($content);
    }

    protected function handleUuidFilter(array &$data, string $currentUuid, string $field, bool $isSearch = false): void
    {
        $contentTypes = ['shared_documents', 'space_documents', 'search'];

        $uuids = [$currentUuid];

        if (($table = type_to_table($field)) === null) {
            return;
        }

        $source = DB::table($table)->where('uuid', $currentUuid)->first();

        $translations = DB::table($table)->where('uuid_host', $source->uuid_host)->where('langcode', '<>', $source->langcode)->get();

        foreach ($translations as $translation) {
            $uuids[] = $translation->uuid;
        }

        foreach ($contentTypes as $contentType) {
            if (!array_key_exists($contentType, $data)) {
                continue;
            }

            if (!isset($data[$contentType]['data']) || empty($data[$contentType]['data'])) {
                continue;
            }

            $contents = [];

            foreach ($data[$contentType]['data'] as $key => $content) {
                if (!array_key_exists($field, $content)) {
                    continue;
                }

                if (isset($content[$field]['data']) && empty($content[$field]['data'])) {
                    if ($isSearch) {
                        unset($data[$contentType][$key]);
                    }

                    continue;
                }

                $references = [];

                foreach ($content[$field]['data'] as $reference) {
                    $references[] = $reference['uuid'];
                }

                $hasCurrentUuid = false;

                foreach ($uuids as $uuid) {
                    if (in_array($uuid, $references)) {
                        $hasCurrentUuid = true;
                        break;
                    }
                }

                if (!$hasCurrentUuid) {
                    if ($isSearch) {
                        unset($data[$contentType]['data'][$key]);
                    }

                    continue;
                }

                $contents[] = $content;
            }

            if (!$isSearch) {
                $data[$contentType]['data'] = $contents;
            }
        }
    }
}
