import { http } from '@inside-library/helpers/Http'

export const fetchSpacesListing = async () => {
  try {
    const { data } = await http.get('content/spaces', {
      fields: ['uuid', 'content_type', 'slug', 'title', 'image', 'admin'],
      filters: {
        'status:eq': 1,
        'is_listed:eq': 1,
        sort: 'weight:asc'
      }
    })

    return data
  } catch (error) {
    throw new Error(error)
  }
}

export const fetchSpaces = async ({ $route }) => {
  try {
    const { data } = await http.get('content/spaces', {
      fields: [
        'uuid',
        'admin',
        'title',
        'slug',
        'content_type',
        'introduction',
        'presentation',
        'banner',
        'content',
        {
          tools: {
            fields: [
              'title',
              'uuid',
              'has_section',
              'tools_link',
              'admin',
              'image',
              'class',
              'content_type',
              'content',
              'icon',
              'icon_color',
              {
                tools_categories: [
                  'uuid',
                  { tools_categories_listings: ['uuid', 'slug'] }
                ]
              }
            ],
            filters: {
              'status:eq': 1,
              limit: 6
            }
          }
        },
        {
          users: {
            fields: [
              'uuid',
              'firstname',
              'lastname',
              'image',
              'function',
              'phone'
            ],
            filters: {
              'status:eq': 1,
              limit: 6
            }
          }
        },
        {
          documents: {
            fields: [
              'uuid',
              'content_type',
              'file',
              'title',
              'admin',
              'published_at',
              'content',
              'icon'
            ],
            filters: {
              'status:eq': 1,
              limit: 6
            }
          }
        },
        {
          news_categories: ['uuid', 'title']
        }
      ],
      filters: { 'slug:eq': $route.params.slug }
    })

    const { data: news } = await http.get('content/news', {
      fields: [
        'uuid',
        'title',
        'slug',
        'image',
        'chapo',
        'content_type',
        'published_at',
        { news_categories: ['uuid', 'title'] }
      ],
      filters: {
        'status:eq': 1,
        sort: 'published_at:desc',
        'published_at:lte': 'now()',
        limit: 4,
        news_categories: {
          'status:eq': 1,
          'published_at:lte': 'now()',
          'uuid:in': data[0].news_categories.data.map(({ uuid }) => uuid)
        }
      }
    })

    return { ...data[0], news }
  } catch (err) {
    throw new Error(err)
  }
}
