<?php

declare(strict_types=1);

namespace Inside\DUVA\Http\Controllers;

use Illuminate\Http\JsonResponse;
use Illuminate\Http\Request;
use Illuminate\Routing\Controller;
use Illuminate\Support\Facades\Auth;
use Inside\Authentication\Models\User;
use Inside\Content\Models\Contents\Modals;
use Inside\Reaction\Facades\Reactions;
use Inside\Reaction\Models\Reaction;
use Symfony\Component\HttpFoundation\Response;
use Symfony\Component\HttpKernel\Exception\NotFoundHttpException;

final class ModalsController extends Controller
{
    public function firstConnection(): JsonResponse
    {
        $user = Auth::user();
        $modal = Modals::query()
            ->where('status', true)
            ->where('first_connection', true)
            ->first();

        if (! $user instanceof User || ! $modal instanceof Modals) {
            throw new NotFoundHttpException('first_connection modal not found');
        }

        /** @var ?Reaction $reaction */
        $reaction = Reaction::query()
            ->where('type', 'first_connection')
            ->where('reactionable_uuid', $modal->uuid)
            ->where('user_uuid', $user->uuid)
            ->orderBy('created_at', 'DESC')
            ->first();

        return response()->json([
            'modal_uuid' => $modal->uuid,
            'reacted' => $reaction !== null,
            'reaction_date' => $reaction?->created_at,
            'reaction_type' => 'first_connection',
        ]);
    }

    public function getReaction(Request $request, string $modalUuid): JsonResponse
    {
        $user = Auth::user();
        $modal = Modals::find($modalUuid);

        if (! $user instanceof User || ! $modal instanceof Modals) {
            throw new NotFoundHttpException('Modal not found');
        }

        $reactionType = $request->query('reaction_type', 'first_connection');

        /** @var ?Reaction $reaction */
        $reaction = Reaction::query()
            ->where('type', $reactionType)
            ->where('reactionable_uuid', $modal->uuid)
            ->where('user_uuid', $user->uuid)
            ->orderBy('created_at', 'DESC')
            ->first();

        return response()->json([
            'reacted' => $reaction !== null,
            'reaction_date' => $reaction?->created_at,
            'reaction_type' => $reactionType,
        ]);
    }

    public function postReaction(Request $request, string $modalUuid): JsonResponse
    {
        $user = Auth::user();
        $modal = Modals::find($modalUuid);

        if (! $user instanceof User || ! $modal instanceof Modals) {
            throw new NotFoundHttpException('Modal not found');
        }

        $reactionType = $request->input('reactionType', 'first_connection');

        $data = [
            'type' => $reactionType,
            'user_uuid' => $user->uuid,
            'reactionable_uuid' => $modal->uuid,
            'reactionable_type' => Modals::class,
        ];

        if (Reaction::where($data)->exists()) {
            Reactions::delete($reactionType, $modal->uuid, $user->uuid);
        }

        $reaction = Reactions::create($data, [], false);

        return response()
            ->json(
                data: [
                    'reacted' => true,
                    'reaction_date' => $reaction['created_at'],
                    'reaction_type' => $reaction['type'],
                ],
                status: Response::HTTP_CREATED
            );
    }
}
