<?php

namespace Inside\DESK\Http\Controllers;

use Illuminate\Http\JsonResponse;
use Illuminate\Http\Request;
use Laravel\Lumen\Routing\Controller;
use Symfony\Component\HttpFoundation\Response;
use Inside\DESK\Facades\SalesForce;
use Illuminate\Support\Facades\Auth;
use Inside\Content\Facades\ContentHelper;

class SalesForceApiController extends Controller
{
    public function login(Request $request): JsonResponse
    {
        try {
            return response()->json(['token' => SalesForce::authenticate()]);
        } catch (\Exception $exception) {
            return response()->json(['message' => $exception->getMessage()], Response::HTTP_BAD_GATEWAY);
        }
    }

    public function getTasks(Request $request): JsonResponse
    {
        $user = Auth::user();
        if (!$user->fra_code) {
            return response()->json(['message' => 'User does not have fracode !'], Response::HTTP_BAD_REQUEST);
        }

        $filters = ContentHelper::extractFiltersInputFromRequest($request);

        try {
            $data = SalesForce::getTasksByCodeFra($user->fra_code, $filters['category'] ?? null);
            if (!empty($data['records'])) {
                $data['records'] = collect($data['records'])->map(function ($record) {
                    $record['ActivityDate'] = is_int($record['ActivityDate']) ? $record['ActivityDate'] : (new \DateTime($record['ActivityDate'] . ' 12:00:00'))->getTimestamp();
                    return $record;
                });
            }
            return response()->json(['data' => $data]);
        } catch (\Exception $exception) {
            return response()->json(['message' => $exception->getMessage()], Response::HTTP_BAD_GATEWAY);
        }
    }

    public function getTask(string $taskId, Request $request): JsonResponse
    {
        try {
            $record = SalesForce::getTask($taskId);
            $record['ActivityDate'] = is_int($record['ActivityDate']) ? $record['ActivityDate'] : (new \DateTime($record['ActivityDate'] . ' 12:00:00'))->getTimestamp();
            return response()->json(['data' => $record]);
        } catch (\Exception $exception) {
            return response()->json(['message' => $exception->getMessage()], Response::HTTP_BAD_GATEWAY);
        }
    }

    public function completeTask(Request $request): JsonResponse
    {
        try {
            $tasks = explode(',', $request->get('tasks'));
            return response()->json(['data' => SalesForce::completeTask($tasks)]);
        } catch (\Exception $exception) {
            return response()->json(['message' => $exception->getMessage()], Response::HTTP_BAD_GATEWAY);
        }
    }

    public function updateTask(string $taskId, Request $request): JsonResponse
    {
        try {
            return response()->json(['data' => SalesForce::updateTask($taskId, $request->getContent())]);
        } catch (\Exception $exception) {
            return response()->json(['message' => $exception->getMessage()], Response::HTTP_BAD_GATEWAY);
        }
    }

    public function getCategories(Request $request): JsonResponse
    {
        try {
            return response()->json(['data' => SalesForce::getCategories()]);
        } catch (\Exception $exception) {
            return response()->json(['message' => $exception->getMessage()], Response::HTTP_BAD_GATEWAY);
        }
    }
}
