<?php

namespace Inside\DELP\Console;

use Illuminate\Console\Command;
use Illuminate\Support\Arr;
use Illuminate\Support\Str;
use Inside\Content\Models\Contents\ExternalPartners;
use Inside\Host\Bridge\BridgeContent;
use League\Csv\Reader;

class ImportExternalPartnersCommand extends Command
{
    /**
     * @var string
     */
    protected $name = 'delp:external:partners:import';

    /**
     * @var string
     */
    protected $signature = 'delp:external:partners:import { --P|path=}';

    /**
     * @var string
     */
    protected $description = 'Import des partenaires externe';

    /**
     * Create a new command instance.
     *
     * @return void
     */
    public function __construct()
    {
        parent::__construct();
    }

    private function getReference(array $record, BridgeContent $bridge, string $bundle, string $column): ?array
    {
        $name = trim(ucwords(strtolower($record[$column])));

        if (empty($name)) {
            return null;
        }

        $query = call_user_func('Inside\\Content\\Models\\Contents\\' . Str::studly($bundle) . '::query');

        $entity = $query->where('title', $name)->first();
        if ($entity) {
            return [$entity->uuid];
        }

        $uuid = $bridge->contentInsert($bundle, [
            'title' => $name,
            'type' => 'node',
            'bundle' => $bundle
        ]);

        if ($uuid) {
            return [$uuid];
        }
        return null;
    }

    private function importExternalPartners(array $externalPartners): void
    {
        $bridge = new BridgeContent();

        foreach ($externalPartners as $partner) {
            if (!trim($partner['entreprise'])) {
                continue;
            }
            $fullNameArray = explode(' ', $partner['nom']);

            $infos = [
                'type' => 'node',
                'bundle' => 'external_partners',
                'title' => trim($partner['entreprise']),
                'partners_comments' => trim($partner['avis/commentaire']),
                'mail' => trim($partner['email']),
                'phone' => trim($partner['telephone']),
                'firstname' => array_shift($fullNameArray),
                'lastname' => implode(' ', $fullNameArray),
                'partners_companies' => $this->getReference($partner, $bridge, 'partners_companies', 'entreprise'),
                'partners_specialities' => $this->getReference($partner, $bridge, 'partners_specialities', 'spécialité'),
                'partners_categories' => $this->getReference($partner, $bridge, 'partners_categories', 'catégorie')
            ];
            $bridge->contentInsert('external_partners', $infos);
        }
    }

    /**
     * import external partner
     */
    public function handle(): void
    {
        $filePath = __DIR__ . '/../../../../../' . env('EXTERNAL_PARTNERS_FILE_PATH', 'imports/external_partners.csv');

        /** @var string|null $path */
        $path = $this->option('path');

        if ($path) {
            $filePath = $path;
        }

        if (!file_exists($filePath)) {
            throw new \Exception('file not found');
        }
        $csv = Reader::createFromPath($filePath, 'r');

        $encoding = mb_detect_encoding($csv->getContent(), mb_list_encodings(), true);

        if ($encoding !== 'UTF-8') {
            $csv->setOutputBOM(Reader::BOM_UTF8);
            $csv->addStreamFilter('convert.iconv.'.$encoding.'/UTF-8');
        }

        $csv->setDelimiter(',');
        $csv->setHeaderOffset(0);

        $headers = array_map(
            function ($item) {
                return trim(strtolower($item));
            },
            $csv->getHeader()
        );

        $externalPartners = iterator_to_array($csv->getRecords($headers));
        $this->importExternalPartners($externalPartners);
    }
}
