<?php

namespace Inside\BELV\Listeners;

use Illuminate\Contracts\Queue\ShouldQueue;
use Illuminate\Support\Collection;
use Inside\Authentication\Models\User;
use Inside\BELV\Helpers\UrlHelper;
use Inside\Content\Events\ContentSynchedEvent;
use Inside\Content\Models\Content;
use Inside\Notify\Listeners\BaseNotificationListener;
use Inside\Notify\Models\NotificationType;

class VacationsUpdatedListener extends BaseNotificationListener implements ShouldQueue
{
    public function handle(ContentSynchedEvent $event): void
    {
        $model = $event->model;
        $fieldName = $event->fieldName;
        $user = $event->user;

        if ($model->content_type != 'vacations' || !$user instanceof User) {
            return;
        }

        if ($fieldName === 'vacations_status') {
            $this->statusUpdated($model, $user);
        }

        if ($fieldName === 'validator') {
            $this->validatorUpdated($model, $user);
        }
    }

    private function validatorUpdated(Content $model, User $user): void
    {
        $targetUser = User::query()->find($model->validator->first()->uuid);

        if (!$targetUser instanceof User || $targetUser->uuid === $user->uuid) {
            return;
        }

        /** @var Collection<NotificationType> $notificationTypes */
        $notificationTypes = NotificationType::where('model', type_to_class($model->content_type))
            ->where('action', 'create')
            ->get();

        $this->route = $targetUser;
        foreach ($notificationTypes as $notificationType) {
            $this->when = $this->getInterval($notificationType);
            $this->notify(
                $notificationType,
                $targetUser,
                $model,
                [
                    'from' => $model->author,
                    'url' => UrlHelper::getModalUrl('validation-conges', $model),
                ]
            );
        }
    }

    private function statusUpdated(Content $model, User $user): void
    {
        $targetUser = User::query()->find($model->author);

        if (!$targetUser instanceof User || $targetUser->uuid === $user->uuid) {
            return;
        }

        $status = $model->vacationsStatus->first()->value;
        $validator = $model->validator->first()->uuid;
        /** @var Collection<NotificationType> $notificationTypes */
        $notificationTypes = NotificationType::where('model', type_to_class($model->content_type))
            ->where('action', 'statusUpdated')
            ->get();
        $this->route = $targetUser;
        foreach ($notificationTypes as $notificationType) {
            $this->when = $this->getInterval($notificationType);
            $this->notify(
                $notificationType,
                $targetUser,
                $model,
                [
                    'from' => $validator,
                    'vacations_status' => $status,
                    'url' => UrlHelper::getModalUrl('gestion-conges', $model),
                ]
            );
        }
    }
}
