<?php

namespace Inside\BELV\Listeners;

use Illuminate\Contracts\Queue\ShouldQueue;
use Inside\Authentication\Models\User;
use Inside\Content\Events\ContentSynchedEvent;
use Inside\Content\Events\ContentUpdatingEvent;
use Inside\Content\Models\Content;
use Inside\Notify\Listeners\BaseNotificationListener;
use Inside\Notify\Models\NotificationType;

class LeadsUpdatedListener extends BaseNotificationListener implements ShouldQueue
{
    public function handle(ContentSynchedEvent|ContentUpdatingEvent $event): void
    {
        $model = $event->model;
        $user = $event->user;

        if (!$model->exists || $model->content_type != 'leads' || !$user instanceof User) {
            return;
        }

        if ($event instanceof ContentSynchedEvent) {
            $fieldName = $event->fieldName;

            if ($fieldName === 'delivery') {
                $this->notifyAuthorDeliveryChanged($model, $user);
                $this->notifyDeliveryAssigned($model, $user);
            }

            if ($fieldName === 'leads_status') {
                $notificationTypes = $this->getNotificationTypes($model, 'statusUpdated');
                $this->notifyAll($user, $notificationTypes, $model, ['from' => $user->uuid]);
            }
        }

        if ($event instanceof ContentUpdatingEvent) {
            $dirty = $model->getDirty();

            if (!empty(array_diff(array_keys($dirty), ['updated_at', 'author', 'created_at']))) { // true si il des champs en plus de 'updated_at' et 'author' sont modifiés.
                $this->notifyAllUpdate($model, $user, $dirty);
            }
        }
    }

    private function notifyAllUpdate(Content $model, User $user, array $updatedFields): void
    {
        if (in_array('amount_invoiced', $updatedFields)) {
            $notificationTypes = $this->getNotificationTypes($model, 'amountInvoicedUpdated');
        } else {
            $notificationTypes = $this->getNotificationTypes($model, 'update');
        }
        $this->notifyAll($user, $notificationTypes, $model, ['from' => $user->uuid]);
    }

    private function notifyAuthorDeliveryChanged(Content $model, User $user): void
    {
        $author = $model->author;
        $delivery = $model->delivery->first()->uuid;
        $notificationTypes = $this->getNotificationTypes($model, 'deliveryUpdated');

        if ($author === $user->uuid || $author === $delivery) {
            return;
        }

        $this->notifyUser($author, $notificationTypes, $model, ['from' => $delivery]);
    }

    private function notifyDeliveryAssigned(Content $model, User $user): void
    {
        $delivery = $model->delivery->first()->uuid;
        $notificationTypes = $this->getNotificationTypes($model, 'deliveryAssigned');

        if ($delivery === $user->uuid) {
            return;
        }

        $this->notifyUser($delivery, $notificationTypes, $model, ['from' => $user->uuid]);
    }

    public function getNotificationTypes(Content $model, string $action): array
    {
        return NotificationType::where('model', type_to_class($model->content_type))
            ->where('action', $action)
            ->get()
            ->toArray();
    }

    private function notifyAll(User $currentUser, array $notificationTypes, Content $model, array $data): void
    {
        $targetUsers = User::whereHas( // Admins cross-selling
            'roles',
            function ($query) {
                return $query->where('inside_roles.name', 'Admin ventes-croisées');
            }
        )->pluck('uuid')->toArray();

        $targetUsers[] = $model->author;
        $targetUsers[] = $model->delivery->first()->uuid;
        $targetUsers = array_unique($targetUsers);
        $targetUsers = array_filter(
            $targetUsers,
            function ($targetUser) use ($currentUser) {
                return $targetUser !== null && $targetUser !== $currentUser->uuid;
            }
        );

        foreach ($targetUsers as $targetUser) {
            $this->notifyUser($targetUser, $notificationTypes, $model, $data);
        }
    }

    private function notifyUser(string $userUuid, array $notificationTypes, Content $model, array $data): void
    {
        /** @var ?User $user */
        $user = User::find($userUuid);
        $this->route = $user;
        foreach ($notificationTypes as $notificationType) {
            $this->when = $this->getInterval($notificationType);
            $this->notify($notificationType, $user, $model, $data);
        }
    }
}
