<?php

namespace Inside\BCLH\Http\Controllers;

use Illuminate\Http\JsonResponse;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Auth;
use Illuminate\Support\Facades\Queue;
use Illuminate\Support\Facades\Storage;
use Inside\Authentication\Models\User;
use Inside\BCLH\Facades\Pdf;
use Inside\BCLH\Jobs\GenerateBasket;
use Inside\BCLH\Jobs\PrepareEbook;
use Inside\BCLH\Services\PdfService;
use Inside\Content\Models\Contents\Brands;
use Inside\Content\Models\Contents\Folders;
use Inside\Content\Models\Contents\Rops;
use Inside\Content\Models\Contents\Sops;
use Inside\Reaction\Models\Reaction;

/**
 * Inside content export controller.
 *
 * @category Class
 * @author   Maecia <technique@maecia.com>
 * @license  http://www.gnu.org/copyleft/gpl.html GNU General Public License
 * @link     http://www.maecia.com/
 */
class ExportController
{
    /**
     * Export a pdf
     *
     * @param Request $request
     * @return JsonResponse
     */
    public function ebook(Request $request): JsonResponse
    {
        $brand = $request->has('brand') ? Brands::query()->find($request->get('brand')) : null;
        $folder = $request->has('folder') ? Folders::query()->find($request->get('folder')) : null;

        /** @var User $user */
        $user = Auth::user();
        Queue::push(new PrepareEbook($user->uuid, $user->langcode, $brand, $folder), '', env('BCLH_EBOOK_QUEUE', 'bclh_ebooks'));

        return response()->json([
            'success' => true,
        ]);
    }

    /**
     * Generate a zip file containing files added to the user basket
     *
     * @param Request $request
     * @return JsonResponse
     */
    public function basket(Request $request): JsonResponse
    {
        /** @var User $user */
        $user = Auth::user();

        $reactions = Reaction::query()
            ->where('type', 'basket')
            ->where('user_uuid', $user->uuid)
            ->get();

        Queue::push(new GenerateBasket($user->uuid, $reactions));

        return response()->json([
            'success' => true,
        ]);
    }

    /**
     * Export an image in request format (image or pdf)
     *
     * @param Request $request
     */
    public function image(Request $request): array
    {
        $type = $request->get('type');
        /** @var User $currentUser */
        $currentUser = Auth::user();
        $filename = 'download/'.($type === 'jpg' ? 'signature' : 'card').'-'.$currentUser->uuid.'.'.$type;

        if ($type === 'jpg') {
            $image = $request->get('image');

            [$imageType, $image] = explode(';', $image);
            [, $image] = explode(',', $image);
            $image = base64_decode($image);

            Storage::disk('protected')->put($filename, $image);
        } else {
            $fonts = [
                [
                    'family' => 'Dala Floda',
                    'weight' => 500,
                    'file' => 'DalaFloda-Medium.ttf',
                ],
                [
                    'family' => 'Campanile',
                    'weight' => 200,
                    'file' => 'Campanile-Light.ttf',
                ],
                [
                    'family' => 'Campanile',
                    'weight' => 400,
                    'file' => 'Campanile-Regular.ttf',
                ],
                [
                    'family' => 'Campanile',
                    'weight' => 700,
                    'file' => 'Campanile-Bold.ttf',
                ],
                [
                    'family' => 'GillSansStd',
                    'weight' => 300,
                    'file' => 'GillSansStd-Light.ttf',
                ],
                [
                    'family' => 'GillSansStd',
                    'weight' => 700,
                    'file' => 'GillSansStdBold.ttf',
                ],
                [
                    'family' => 'Metric',
                    'weight' => 600,
                    'file' => 'Metric-Semibold.ttf',
                ],
                [
                    'family' => 'Metric',
                    'weight' => 400,
                    'file' => 'Metric-Regular.ttf',
                ],
                [
                    'family' => 'Minion Pro',
                    'weight' => 400,
                    'file' => 'MinionPro-Regular.ttf',
                ],
                [
                    'family' => 'Knockout-28JuniorFeathrwt',
                    'weight' => 400,
                    'file' => 'Knockout-28JuniorFeathrwt.ttf',
                ],
                [
                    'family' => 'Knockout-31JuniorMiddlewt',
                    'weight' => 400,
                    'file' => 'Knockout-31JuniorMiddlewt.ttf',
                ],
                [
                    'family' => 'Knockout-51Middleweight',
                    'weight' => 400,
                    'file' => 'Knockout-51Middleweight.ttf',
                ],
                [
                    'family' => 'Knockout-32JuniorCruisewt',
                    'weight' => 400,
                    'file' => 'Knockout-32.ttf',
                ],
                [
                    'family' => 'Knockout-33JuniorHeviwt',
                    'weight' => 400,
                    'file' => 'Knockout-33.ttf',
                ],
                [
                    'family' => 'Knockout-49Liteweight',
                    'weight' => 400,
                    'file' => 'Knockout-49.ttf',
                ],
                [
                    'family' => 'Overpass',
                    'weight' => 200,
                    'file' => 'Overpass-Light.ttf',
                ],
                [
                    'family' => 'Overpass',
                    'weight' => 400,
                    'file' => 'Overpass-Regular.ttf',
                ],
                [
                    'family' => 'Overpass',
                    'weight' => 700,
                    'file' => 'Overpass-Bold.ttf',
                ],
                [
                    'family' => 'Reso',
                    'weight' => 400,
                    'file' => 'Reso-Light.ttf',
                ],
                [
                    'family' => 'Reso',
                    'weight' => 700,
                    'file' => 'Reso-SemiBold.ttf',
                ],
                [
                    'family' => 'Rubrik',
                    'weight' => 700,
                    'file' => 'Rubrik-Bold.ttf',
                ],
                [
                    'family' => 'Rubrik',
                    'weight' => 200,
                    'file' => 'Rubrik-Light.ttf',
                ],
                [
                    'family' => 'Rubrik',
                    'weight' => 400,
                    'file' => 'Rubrik-Medium.ttf',
                ],
            ];

            Pdf::createPdf('card', $filename, [
                'fonts' => $fonts,
                'width' => $request->get('width'),
                'height' => $request->get('height'),
                'page1' => $request->get('page1'),
                'page2' => $request->get('page2'),
            ], true, 300);
        }

        return [
            'path' => Storage::disk('protected')->url($filename).'?'.time(),
        ];
    }
}
