<?php

namespace Inside\APRO\Http\Middlewares;

use Closure;
use Illuminate\Http\Request;
use Symfony\Component\HttpFoundation\Response;

class PermissionsFilteredByRoleMiddleware
{
    private array $PATHS = [
        'api/v1/search',
        'api/v1/content/companies',
        'api/v1/form/company_documents'
    ];

    /**
     * @param Request $request
     * @param Closure $next
     * @return mixed
     */
    public function handle(Request $request, Closure $next): mixed
    {
        if (
            !str($request->path())->contains($this->PATHS)
        ) {
            return $next($request);
        }

        if ($request->getMethod() !== 'GET') {
            return $next($request);
        }

        $user = $request->user();

        if (is_null($user) || $user->hasRole('super_administrator')) {
            return $next($request);
        }

        $companies = $user->companies->pluck('company_id')->toArray();

        $companyFolders = $user->companies->map(
            fn ($company) => $company->reverseCompanyFolders->pluck('uuid')->first()
        )->toArray();

        if (is_null($companies)) {
            return $next($request);
        }

        $filtersRaw = $request->query('filters', '{}');
        $filters = is_array($filtersRaw) ?
            $filtersRaw :
            json_decode(is_string($filtersRaw) ? $filtersRaw : '{}', true);
        if (!is_array($filters)) {
            $filters = [];
        }

        $filters['company_id:in'] = $companies;
        $this->setFilters($request, $filters);

        $endpoint = str($request->path());
        $response = $next($request);

        if ($endpoint->contains('api/v1/search')) {
            $content = $this->search($request, $response, $companies);
            set_response($response, $content);
        }

        if ($endpoint->contains('api/v1/form/company_documents')) {
            $content = $this->form($request, $response, $companyFolders);
            set_response($response, $content);
        }

        return $response;
    }

    private function setFilters(Request $request, array $filters): void
    {
        $request->query->set(
            'filters',
            json_encode($filters, JSON_UNESCAPED_SLASHES | JSON_UNESCAPED_UNICODE)
        );
    }

    private function search(Request $request, Response $response, array $companies): array
    {
        $content = json_decode_response($response);

        if (collect($content['data'])->isEmpty()) {
            return $content;
        }

        if (!in_array($content['data'][0]['content_type'], [
            'companies',
        ])) {
            return $content;
        }

        $filteredData = [];
        foreach ($content['data'] as $index => $data) {
            $companyId = data_get($data, 'company_id', "");
            if (in_array($companyId, $companies)) {
                $filteredData[] = $data;
            }
        }
        $content['data'] = $filteredData;

        return $content;
    }

    private function form(Request $request, Response $response, array $companyFoldersUuids): array
    {
        $content = json_decode_response($response);

        if (collect($content['data'])->isEmpty()) {
            return $content;
        }

        $companyFolders = array_filter(
            data_get($content, 'data', []),
            fn ($data) => $data['name'] == 'company_folders'
        );

        $formKey = key($companyFolders);
        $companyFolders = $companyFolders[$formKey];

        $allowedCompanyFolders = data_get($companyFolders, 'allowed_values.company_folders.fr', []);

        if (collect($allowedCompanyFolders)->isEmpty()) {
            return $content;
        }

        $filteredData = array_filter($allowedCompanyFolders, fn ($companyFolder) => in_array($companyFolder['uuid'], $companyFoldersUuids));

        $content['data'][$formKey]['allowed_values']['company_folders']['fr'] = array_values($filteredData);
        $content['data'][$formKey]['options']['allowed_values']['company_folders']['fr'] = array_values($filteredData);

        return $content;
    }
}
