<?php

declare(strict_types=1);

use Illuminate\Support\Arr;
use Illuminate\Support\Collection;
use Illuminate\Support\Str;
use Inside\Authentication\Models\User;
use Inside\Content\Models\Contents\Departments;
use Inside\Content\Models\Contents\Functions;
use Inside\Content\Models\Contents\Locations;
use Inside\Content\Models\Contents\Users;
use Inside\Host\Bridge\BridgeContent;
use Inside\Import\Models\UserReference;

if (! function_exists('get_content_bridge')) {
    function get_content_bridge(): BridgeContent
    {
        static $bridge = null;

        if ($bridge === null) {
            $bridge = new BridgeContent();
        }

        return $bridge;
    }
}

return [
    'sync_attributes' => [
        'mail' => 'mail',
        'name' => 'displayName',
        'lastname' => 'surname',
        'firstname' => 'givenName',
        'functions' => 'jobTitle',
        'locations' => 'city',
        'phone' => 'businessPhones',
        'mobile_phone' => 'mobilePhone',
        'departments' => 'department',
        'users' => static function ($user, BridgeContent $bridge): ?string {
            if (! isset($user->manager)) {
                return null;
            }

            try {
                /** @var UserReference $manager */
                $manager = UserReference::where([
                    'provider_name' => 'default',
                    'provider_type' => 'azuread',
                    'reference' => $user->manager,
                ])->firstOrFail();
                /** @var User $manager */
                $manager = User::findOrFail($manager->user_id);
                if ($manager->status === true) {
                    return $manager->uuid;
                }
            } catch (Throwable) {
            }

            return null;
        },
        'status' => static fn (): bool => true,
    ],
    'postfilter' => static function (Collection $users): Collection {
        $filteredUsers = $users->filter(fn (mixed $user) => $user->givenName && $user->surname && $user->mail && $user->department !== 'ROBOT');

        if ($filteredUsers->isEmpty()) {
            return $filteredUsers;
        }

        $bridge = get_content_bridge();

        $departmentsToImport = $filteredUsers->pluck('department')->filter()->unique();

        // @phpstan-ignore-next-line
        $departments = Departments::pluck('uuid', 'title');

        $departmentsToImport
            ->diff($departments->keys())
            ->each(fn (string $department) => $departments[$department] = $bridge->contentInsert('departments', ['title' => $department]));

        $locationsToImport = $filteredUsers
            ->pluck('city')
            ->filter()
            ->unique();

        // @phpstan-ignore-next-line
        $locations = Locations::pluck('uuid', 'title');

        $locationsToImport
            ->diff($locations->keys())
            ->each(fn (string $location) => $locations[$location] = $bridge->contentInsert('locations', ['title' => $location]));

        // @phpstan-ignore-next-line
        $managerUsers = Users::pluck('uuid', 'email');

        $functionsToImport = $filteredUsers->pluck('jobTitle')->filter()->unique();

        // @phpstan-ignore-next-line
        $functions = Functions::pluck('uuid', 'title');

        $functionsToImport
            ->diff($functions->keys())
            ->each(fn (string $function) => $functions[$function] = $bridge->contentInsert('functions', ['title' => $function]));

        return $filteredUsers->map(function (mixed $user) use ($departments, $locations, $managerUsers, $functions) {
            $user->department = $departments->get($user->department, null);
            $user->city = $locations->get($user->city, null);
            $user->givenName = (string) str($user->givenName)->lower()->title();
            $user->surname = (string) str($user->surname)->lower()->title();
            $user->mail = (string) str($user->mail)->lower();
            $user->gestionnairenom = $managerUsers->get($user->gestionnairenom, null);
            $user->jobTitle = $functions->get($user->jobTitle, null);

            $user->businessPhones = Arr::first($user->businessPhones);
            if (Str::contains($user->businessPhones, '-')) {
                [$user->businessPhones] = explode('-', $user->businessPhones, 2);
            }
            if (is_string($user->businessPhones)) {
                $user->businessPhones = trim($user->businessPhones);
            }
            if (is_string($user->mobilePhone)) {
                $user->mobilePhone = trim($user->mobilePhone);
            }

            return $user;
        });
    },
    'custom_attributes' => [
        'manager' => 'userPrincipalName',
    ],
];
