<?php

use Adldap\Models\Entry;
use Inside\Host\Bridge\BridgeContent;
use Inside\Content\Models\Contents\Residencies;

if (!function_exists('get_territorial_groupings')) {
    function get_territorial_groupings(Entry $user, BridgeContent $bridge): array
    {
        $return  = [];
        $residenciesString = $user->getAttribute('info')[0];
        if (!$residenciesString) {
            return $return;
        }
        $residencesCodes = explode(";", $residenciesString);
        $territories = collect($residencesCodes)
            ->map(
                fn (string $residenciesCode) => reference('residencies')
                    ->identifiedBy("code")
                    ->from($residenciesCode)
                    ->with('territorialGroupings')
                    ->withoutCreation()
                    ->getModels()
                    ->first()?->territorialGroupings->first()->title
            )->unique();

        $territories->each(function (?string $territory) use (&$return, $bridge) {
            if (empty($territory)) {
                return;
            }

            $existingUuid = type_to_class('territorial_groupings')::where('code', $territory)->first()?->uuid;
            if (is_null($existingUuid)) {
                $existingUuid = type_to_class('territorial_groupings')::where('title', $territory)->first()?->uuid;
                if ($existingUuid) {
                    $bridge->contentUpdate('territorial_groupings', [
                        'type' => 'node',
                        'bundle' => 'territorial_groupings',
                        'code' => $territory,
                        'uuid' => $existingUuid,
                    ]);
                }
            }



            $return[] = $existingUuid ?? $bridge->contentInsert('territorial_groupings', [
                'type' => 'node',
                'bundle' => 'territorial_groupings',
                'title' => $territory,
                'code' => $territory,
            ]);
        });
        return $return;
    }
}

if (!function_exists('get_ldap_reference')) {
    function get_ldap_reference(Entry $user, BridgeContent $bridge, string $attribute, string $contentType): ?array
    {
        if (!$user->hasAttribute($attribute)) {
            return null;
        }

        $uuids = [];
        $titles = $user->getAttribute($attribute);

        foreach ($titles as $title) {
            $entity = call_user_func(type_to_class($contentType).'::query')
                ->where('title', $title)
                ->first();

            if ($entity) {
                $uuid = $entity->uuid;
            } else {
                $uuid = $bridge->contentInsert(
                    $contentType,
                    [
                        'type' => 'node',
                        'bundle' => $contentType,
                        'title' => $title,
                        'uid' => 1,
                    ]
                );
            }

            $uuids[] = $uuid;
        }

        return empty($uuids) ? null : $uuids;
    }
}


return [
    'usernames' => [
        'ldap' => [
            'discover' => env('ADLDAP_LDAP_DISCOVER', 'mail'),
            'authenticate' => env('ADLDAP_LDAP_AUTHENTICATE', 'mail'),
        ],
        'eloquent' => env('ADLDAP_ELOQUENT_VALUE', 'email'),
    ],
    'discovering_attributes' => 'name',
    'sync_attributes' => [
        'mail' => 'userprincipalname',
        'is_maintenance' => function () {
            return false;
        },
        'name' => 'samaccountname',
        'firstname' => 'givenname',
        'lastname' => 'sn',
        'fullname' => 'cn',
        'mobile_phone' => fn (Entry $user) => $user->getAttribute('mobile')[0],
        'function' => 'title',
        'services' => fn (Entry $user, BridgeContent $bridgeContent) => get_ldap_reference($user, $bridgeContent, 'department', 'services'),
        'territorial_groupings' => fn (Entry $user, BridgeContent $bridgeContent) => get_territorial_groupings($user, $bridgeContent),
        'residencies' => function (Entry $user) {
            $residenciesString = $user->getAttribute('info')[0];
            if (!$residenciesString) {
                return null;
            }
            $residencies = explode(";", $residenciesString);
            $uuids = [];
            foreach ($residencies as $residency) {
                $residencyContent = Residencies::query()->where('code', $residency)->first();
                if ($residencyContent->uuid) {
                    $uuids[] = $residencyContent->uuid;
                }
            }

            return $uuids;
        },
    ],
];
