<?php

declare(strict_types=1);

namespace Inside\ALJT\Services;

use Illuminate\Support\Str;
use Inside\ALJT\Contracts\ContentsCsvImporter as ContentCsvImporterContract;
use Inside\Host\Bridge\BridgeContent;
use League\Csv\ByteSequence;
use League\Csv\CharsetConverter;
use League\Csv\Reader;

class ContentsCsvImporter implements ContentCsvImporterContract
{
    private ?BridgeContent $bridge = null;

    public function getBridge(): BridgeContent
    {
        return $this->bridge ??= new BridgeContent();
    }

    public function getCsvRecords(string $filePath, string $delimiter): array
    {
        $csv = Reader::createFromPath($filePath);
        $encoding = mb_detect_encoding($csv->toString(), mb_list_encodings(), true);

        if ($encoding !== 'UTF-8') {
            $csv->setOutputBOM(ByteSequence::BOM_UTF8);
            $csv->addStreamFilter(CharsetConverter::getFiltername((string) $encoding, 'UTF-8'));
        }

        $csv->setDelimiter($delimiter);
        $csv->setHeaderOffset(0);

        $headers = array_map(
            fn (string $item) => Str::lower(trim($item)),
            $csv->getHeader()
        );

        return iterator_to_array($csv->getRecords($headers));
    }

    public function formatRecord(array $record, string $contentType): array
    {
        $syncAttributes = config("contents_import.$contentType.sync_attributes", []);

        $formattedRecord = [];
        foreach ($syncAttributes as $modelField => $recordField) {
            if (!isset($record[$recordField])) {
                continue;
            }

            if (is_callable($recordField)) {
                $formattedRecord[$modelField] = $recordField($record);
                continue;
            }

            $formattedRecord[$modelField] = $record[$recordField];
        }

        return $formattedRecord;
    }

    public function importContent(array $data, string $contentType): ?string
    {
        $identifier = config("contents_import.$contentType.identifier", 'title');

        if (empty($data[$identifier])) {
            return null;
        }

        $existingUuid = type_to_class($contentType)::where($identifier, $data[$identifier])->first()?->uuid;

        if ($existingUuid !== null) {
            $data['uuid'] = $existingUuid;
        }

        $data['territorial_groupings'] = $this->insertRelation('territorial_groupings', $data['territorial_groupings']);
        $data['residencies_types'] = $this->insertRelation('residencies_types', $data['residencies_types']);

        return $this->getBridge()->contentInsert(
            type: $contentType,
            data: $data
        );
    }

    public function insertRelation(string $contentType, string $title): ?string
    {
        if (!$title) {
            return null;
        }

        $data['title'] = $title;
        $existingUuid = type_to_class($contentType)::where('title', $title)->first()?->uuid;
        if ($existingUuid !== null) {
            return $existingUuid;
        }

        return $this->getBridge()->contentInsert(
            type: $contentType,
            data: $data
        );
    }
}
