<?php

declare(strict_types=1);

namespace Inside\ALJT\Console;

use Illuminate\Console\Command;
use Illuminate\Support\Facades\File;
use Inside\ALJT\Contracts\ContentsCsvImporter;

class ContentsCsvImportCommand extends Command
{
    /**
     * @var string
     */
    protected $name = 'aljt:content:import';

    /**
     * @var string
     */
    protected $signature = 'aljt:content:import {type}
                                                {--d|delimiter=,}
                                                {--p|path=}';

    /**
     * @var string
     */
    protected $description = 'Import contents';

    public function handle(ContentsCsvImporter $contentsCsvImporter): void
    {
        /** @var string $type */
        $type = $this->argument('type');

        /** @var string $delimiter */
        $delimiter = empty($this->option('delimiter')) ? ',' : $this->option('delimiter');

        /** @var string $filePath */
        $filePath = empty($this->option('path')) ? __DIR__.'/../../../../../storage/app/csv/'.$type.'.csv' : $this->option('path');

        if (! config("contents_import.$type")) {
            $this->error("Configuration not found for the type $type");
            return;
        }

        if (! File::exists($filePath)) {
            $this->error('File not found');
            return;
        }

        $this->getOutput()->writeln("<comment>Début de l'import $type !</comment>");

        $records = $contentsCsvImporter->getCsvRecords($filePath, $delimiter);

        $bar = $this->getOutput()->createProgressBar(count($records));
        $bar->setFormat("%message%\n %current%/%max% [%bar%] %percent:3s%%");

        $imported = collect();
        foreach ($records as $record) {
            $data = $contentsCsvImporter->formatRecord($record, $type);
            $bar->setMessage("Importation du contenu => {$data['title']}");

            if ($uuid = $contentsCsvImporter->importContent($data, $type)) {
                $imported->push($uuid);
            }

            $bar->advance();
        }

        $bar->finish();
        $this->getOutput()->newLine();
        $this->getOutput()->writeln("<comment>Import terminé: {$imported->count()} contenus importés.</comment>");
    }
}
