<?php

declare(strict_types=1);

use Inside\Host\Bridge\BridgeContent;

function get_entity_reference(?string $value, BridgeContent $bridge, string $contentType, ?string $identifier = 'title', bool $create = true): ?string
{
    if (empty($value)) {
        return null;
    }

    $existingUuid = type_to_class($contentType)::where($identifier, $value)->first()?->uuid;

    if (! $create && $existingUuid === null) {
        return null;
    }

    return $existingUuid ?? $bridge->contentInsert($contentType, [
        'type' => 'node',
        'bundle' => $contentType,
        $identifier => $value,
    ]);
}

function get_territorial_groupings_csv(array $record, BridgeContent $bridge): array
{
    $return = [];
    $residencesCodes = explode(';', $record['résidences']);

    $territories = collect($residencesCodes)
        ->map(
            fn (string $residenciesCode) => reference('residencies')
            ->identifiedBy("code")
            ->from($residenciesCode)
            ->with('territorialGroupings')
            ->withoutCreation()
            ->getModels()
            ->first()?->territorialGroupings->first()->title
        )->unique();

    $territories->each(function (?string $territory) use (&$return, $bridge) {
        if (!$territory) {
            return;
        }

        $existingUuid = type_to_class('territorial_groupings')::where('code', $territory)->first()?->uuid;

        if (!$existingUuid) {
            $existingUuid = type_to_class('territorial_groupings')::where('title', $territory)->first()?->uuid;
            if ($existingUuid) {
                $bridge->contentUpdate('territorial_groupings', [
                    'type' => 'node',
                    'bundle' => 'territorial_groupings',
                    'code' => $territory,
                ]);
            }
        }



        $return[] = $existingUuid ?? $bridge->contentInsert('territorial_groupings', [
            'type' => 'node',
            'bundle' => 'territorial_groupings',
            'title' => $territory,
            'code' => $territory,
        ]);
    });
    return $return;
}

return [
    'files' => [
        'aljt' => [
            'path' => env('CSV_USERS_PATH', __DIR__.'/../../../../imports/import.csv'),
            'separator' => env('CSV_USERS_SEPARATOR', ','),
            'fields' => [
                'name' => fn (array $record) => trim($record['mail']),
                'mail' => fn (array $record) => trim($record['mail']),
                'firstname' => fn (array $record) => trim($record['prénom']),
                'lastname' => fn (array $record) => trim($record['nom']),
                'function' => fn (array $record) => trim($record['fonction']),
                'phone' => fn (array $record) => trim($record['téléphone']),
                'mobile_phone' => fn (array $record) => trim($record['portable']),
                'services' => fn (array $record, BridgeContent $bridgeContent) => get_entity_reference(
                    value: trim($record['service']),
                    bridge: $bridgeContent,
                    contentType: 'services'
                ),
                'territorial_groupings' => fn (array $record, BridgeContent $bridgeContent) => get_territorial_groupings_csv($record, $bridgeContent),
                'residencies' => fn (array $record, BridgeContent $bridgeContent) => collect(explode(';', $record['résidences']))
                    ->map(fn (string $residenciesCode) => get_entity_reference(
                        value: trim($residenciesCode),
                        bridge: $bridgeContent,
                        contentType: 'residencies',
                        identifier: 'code',
                        create: false
                    ))
                    ->filter()
                    ->values()
                    ->all(),
                'status' => fn () => 1,
            ],
        ],
    ],
];
