<?php

use Illuminate\Support\Carbon;
use Illuminate\Support\Collection;
use Illuminate\Support\Facades\Auth;
use Inside\Content\Models\Contents\Cities;
use Inside\Support\Str;
use Inside\Content\Models\Contents\Departments;
use Inside\Content\Models\Contents\Offers;
use Inside\Content\Models\Contents\Options;
use Inside\Host\Bridge\BridgeContent;

return [
    'files' => [
        'agre' => [
            'path' => env("AGRE_USERS_PATH", storage_path('../imports/agencies_import.csv')),
            'separator' => ',',
            'fields' => [
                'mail' => 'adresse e-mail',
                'name' => 'nom',
                'firstname' => static fn ($record): string => $record['nom'],
                'lastname' => static fn ($record): string => $record['nom'],
                'cities' => static fn ($record): string => $record['ville'],
                'departments' => static fn ($record): string => $record['département'],
                'address' => static fn ($record): string => $record['adresse'],
                'phone' => static fn ($record): string => $record['téléphone'],
                'website' => static fn ($record): string => $record['site web'],
                'business_software' => static fn ($record): string => $record['logiciel métier'],
                'options' => static fn ($record): array => $record['options'],
                'offers' => static fn ($record): string => $record['formule club'],
                'send_mail' => static fn (): bool => false,
                'status' => static fn (): bool => true,
            ],
            'original_fields' => [
                'adresse e-mail' => 'mail',
                'nom' => 'name',
                'ville'  => 'city',
                'département'  => 'word',
                'adresse'  => 'address',
                'site web'  => 'website',
                'logiciel métier' => 'word',
                'options' => 'word',
                'formule club' => 'word'
            ],
        ],
    ],
    'postfilter' => static function (Collection $users): array {
        $bridge = new BridgeContent();

        $options = Options::query()
            ->get(['title', 'uuid'])
            ->transform(fn ($option) => [
                'uuid' => $option->uuid,
                'title' => Str::of($option->title)->replace(' ', '')->lower()
            ]);

        $departments = Departments::query()
            ->get(['title', 'uuid'])
            ->transform(fn ($department) => [
                'uuid' => $department->uuid,
                'title' => Str::before($department->title, '-')
            ]);

        $offers = Offers::query()
            ->get(['title', 'uuid'])
            ->transform(fn ($offer) => [
                'uuid' => $offer->uuid,
                'title' => Str::of($offer->title)->lower()->slug()
            ])->pluck('uuid', 'title');

        $cities = Cities::query()->get()->pluck('uuid', 'title')->toArray();

        $users->transform(function ($user) use ($departments, $options, &$cities, $offers, $bridge) {
            /**
             *  Filter on "tags" to match with our "Option Content Type"
             */
            $user['options'] = collect(explode(',', $user['tags']))
                ->map(
                    fn ($tag) => (string) Str::of($tag)
                    ->ascii()
                    ->after('-')
                    ->replace(' ', '')
                    ->lower()
                )
                ->map(fn ($option) => $options->firstWhere('title', $option)['uuid'])
                ->filter()
                ->toArray();

            $department = $departments->firstWhere('title', Str::of($user['département']));

            if ($department !== null) {
                $user['département'] = $department['uuid'];
            }



            /**
             *  Sanitizing other fields
             */
            $city = Str::upper($user['ville']);

            if ($city !== '' && !array_key_exists($city, $cities)) {
                $cities[$city] = $bridge->contentInsert('cities', [
                    'title' => $city,
                    'status' => 1,
                    'author' => Auth::user()?->uuid,
                    'langcode' => 'fr',
                    'published_at' => Carbon::now(),
                ]);
            }

            if ($user['site web'] !== '' && !Str::startsWith($user['site web'], 'http')) {
                $user['site web'] = "https://" . $user['site web'];
            }

            if ($user['formule club'] !== '') {
                $user['formule club'] = $offers[Str::of($user['formule club'])->lower()->slug()->__toString()];
            }

            $user['nom'] = Str::upper($user['nom']);
            $user['ville'] = $cities[$city];

            return $user;
        });

        return $users->toArray();
    }
];
