<?php

namespace Inside\AGFR\Services;

use Illuminate\Database\QueryException;
use Illuminate\Support\Facades\DB;
use Illuminate\Support\Facades\Log;
use Inside\Permission\Facades\Permission;
use Inside\Permission\Services\PermissionHelperService;

class PermissionCustomFoldersService
{
    private PermissionHelperService $permissionHelper;

    public function __construct()
    {
        $this->permissionHelper = PermissionHelperService::getInstance();
    }

    /**
     * Attribute specific permissions to franchised spaces folders $uuids to let $roles read it !
     *
     * @param string $uuid
     * @param array $roleToBeGrantedIds
     * @param array $roleToBeRejectedIds
     */
    public function setPermissions(string $uuid, array $roleToBeGrantedIds, array $roleToBeRejectedIds = []): void
    {
        $permissionSchema = [
            'authorizable_type' => type_to_class('franchised_spaces_folders'),
            'authorizable_uuid' => $uuid,
            'action' => 'read',
            'invert' => false,
            'children' => true,
        ];
        // grant
        foreach ($roleToBeGrantedIds as $roleId) {
            $permissionSchemaID = $this->permissionHelper->getOrCreatePermissionSchema($permissionSchema);

            $this->grantPermissionSchemaToRoleId($permissionSchemaID, $roleId);
            Permission::buildPermissionForRole($roleId);
        }
        // reject
        $permissionSchema['invert'] = true;
        foreach ($roleToBeRejectedIds as $roleId) {
            $permissionSchemaID = $this->permissionHelper->getOrCreatePermissionSchema($permissionSchema);

            $this->grantPermissionSchemaToRoleId($permissionSchemaID, $roleId);
            Permission::buildPermissionForRole($roleId);
        }
    }

    /**
     * Grant permission schema $schemaId to $roleId
     *
     * @param int $schemaId
     * @param int $roleId
     */
    protected function grantPermissionSchemaToRoleId(int $schemaId, int $roleId): void
    {
        try {
            DB::table('inside_roles_permissions_schema')->insert(
                [
                    'role_id' => $roleId,
                    'permission_schema_id' => $schemaId,
                    'is_content_specific' => true,
                ]
            );
        } catch (QueryException $e) {
            // Do nothing
            Log::error('[FranchisedSpacesListeners] Failed to grant permission schema <'.$schemaId.'> to role <'.$roleId.'>');
        }
    }
}
