<?php

use Illuminate\Support\Facades\Storage;
use Illuminate\Support\Str;
use Inside\Host\Bridge\BridgeContent;

if (!function_exists('get_entity_reference')) {
    function get_entity_reference(array $record, BridgeContent $bridge, string $bundle, string $column, string $field = 'title'): ?array
    {
        $name = trim($record[$column]);
        if (empty($name) || $name === 'Null') {
            return null;
        }

        $values = null;
        if (Str::contains($record[$column], ",")) {
            $values = explode(",", $record[$column]);
        }

        $query = call_user_func(type_to_class($bundle) . '::query');

        if (is_array($values)) {
            $dataUuid = [];
            foreach ($values as $value) {
                if (empty($value)) {
                    continue;
                }
                $entity = $query->where($field, trim($value))->first();
                if ($entity) {
                    $dataUuid[] = $entity->uuid;
                } else {
                    $uuidReference = insert_reference(trim($value), $bundle, $bridge);
                    $dataUuid[] = $uuidReference;
                }
            }

            return $dataUuid;
        } else {
            $entity = $query->where($field, $name)->first();
            if ($entity) {
                return [$entity->uuid];
            }
        }

        return null;
    }
}

if (!function_exists('insert_reference')) {
    function insert_reference(string $name, string $bundle, BridgeContent $bridge): ?string
    {
        $uuidBridge = $bridge->contentInsert($bundle, [
            'title' => $name,
            'type' => 'node',
            'bundle' => $bundle,
        ]);

        if ($uuidBridge) {
            return $uuidBridge;
        }

        return null;
    }
}

if (!function_exists('get_default_image')) {
    function get_default_image(string $imgName): string
    {
        $imgPath = 'franchised_spaces/'.$imgName;

        $chunkId = Str::random(32);
        $finalFolderPath = "chunks/$chunkId";
        $finalPath = "$finalFolderPath/$imgName";
        Storage::makeDirectory($finalFolderPath);
        Storage::copy($imgPath, $finalPath);

        return $finalPath;
    }
}

return [
    'franchised_spaces' => [
        'fields' => [
            'type' => function () {
                return 'node';
            },
            'bundle' => function () {
                return 'franchised_spaces';
            },
            'title' => function ($record) {
                return trim($record['raison sociale']);
            },
            'sectors' => function ($record) {
                return trim($record['secteur (contrat)']);
            },
            'zip' => function ($record) {
                return trim($record['cp']);
            },
            'city' => function ($record) {
                return trim($record['ville']);
            },
            'departments' => function ($record, $bridge) {
                return get_entity_reference($record, $bridge, 'departments', 'departement');
            },
            'franchised' => function ($record, $bridge) {
                return get_entity_reference($record, $bridge, 'users', 'nom', 'lastname');
            },
            'image' => function () {
                return get_default_image('image-listing.jpg');
            },
            'banner' => function () {
                return get_default_image('banner.jpg');
            },
        ],
    ],
];
