<?php

namespace Inside\AGFR\Http\Middlewares;

use Closure;
use Illuminate\Http\Request;
use Illuminate\Support\Collection;
use Inside\Authentication\Models\User;
use Symfony\Component\HttpFoundation\Response;

class UsersDirectoryPermissionMiddleware
{
    private const SEAT = 'seat';
    private const FRANCHSED = 'franchised';

    private array $match = [
        '#^api/v1/content/users$#',
        '#^api/v1/search$#'
    ];

    /**
     * @param Request $request
     * @param Closure $next
     * @return mixed
     */
    public function handle(Request $request, Closure $next): mixed
    {
        if ($request->getMethod() !== 'GET') {
            return $next($request);
        }

        $matchedDirectoryPath = false;
        foreach ($this->match as $match) {
            $matches = [];
            if (preg_match($match, $request->path(), $matches)) {
                $matchedDirectoryPath = true;
            }
        }

        if (!$matchedDirectoryPath) {
            return $next($request);
        }


        $user = $request->user();

        if (is_null($user) || $user->hasRole('super_administrator')) {
            return $next($request);
        }

        $userStatusMachineName = $user->usersStatus->pluck('name_machine')->first();
        $userFranchisedSpaceUuids = $user->franchisedSpaces->pluck('uuid')->toArray();

        if (self::SEAT === $userStatusMachineName) {
            return $next($request);
        }


        $filters = $request->query->get('filters');
        $arrayFilters = !empty($filters) && is_string($filters) ? json_decode($filters, true) : $filters;
        if (!is_array($arrayFilters)) {
            $arrayFilters = [];
        }
        if (isset($arrayFilters['uuid:eq']) ||isset($arrayFilters['uuid'])) {
            return $next($request);
        }

        if (isset($arrayFilters['global'])) {
            return $next($request);
        }
        $statuses = type_to_class('users_status')::all()->pluck('name_machine', 'uuid');

        $userStatusSearchableKey =  'users_status';
        $franchisedSpacesSearchableKey = 'franchised_spaces';

        $unrestrictedStatuses = self::FRANCHSED === $userStatusMachineName
            ? $statuses->filter(fn ($status) => in_array($status, [self::FRANCHSED, self::SEAT]))->keys()->toArray()
            : $statuses->filter(fn ($status) => in_array($status, [self::SEAT]))->keys()->toArray();

        $hasOrFilter = false;
        if (isset($arrayFilters[$userStatusSearchableKey])) {
            if (collect($unrestrictedStatuses)->contains(collect($arrayFilters[$userStatusSearchableKey])->first())) {
                return $next($request);
            } else {
                $arrayFilters[$franchisedSpacesSearchableKey]['uuid:eq'] = $userFranchisedSpaceUuids[0];
            }
        } else {
            $hasOrFilter = $request->path() === 'api/v1/search';
            $arrayFilters['$or'][$userStatusSearchableKey]['uuid:in'] = $unrestrictedStatuses;
            $arrayFilters['$or'][$franchisedSpacesSearchableKey]['uuid:in'] = $userFranchisedSpaceUuids;
        }


        $request->query->set(
            'filters',
            json_encode($arrayFilters, JSON_UNESCAPED_SLASHES | JSON_UNESCAPED_UNICODE)
        );

        return $hasOrFilter ? $this->filterSearchResult($next($request), $unrestrictedStatuses, $userFranchisedSpaceUuids) : $next($request);
    }

    public function filterSearchResult(mixed $response, array $unrestrictedStatuses, array $userFranchisedSpaceUuids): mixed
    {
        $content = json_decode_response($response);

        if (collect($content['data'])->isEmpty()) {
            return $response;
        }

        $allowedStatuses = array_map('strval', $unrestrictedStatuses);
        $allowedSpaces   = array_map('strval', $userFranchisedSpaceUuids);

        $filteredUsers = collect($content['data'])
            ->filter(function (array $user) use ($allowedStatuses, $allowedSpaces) {
                $statusUuid     = data_get($user, 'users_status.uuid');
                $franchiseUuid  = data_get($user, 'franchised_spaces.uuid');

                $matchesStatus   = is_string($statusUuid) && in_array($statusUuid, $allowedStatuses, true);
                $matchesFranchise= is_string($franchiseUuid) && in_array($franchiseUuid, $allowedSpaces, true);

                return $matchesStatus || $matchesFranchise;
            })
            ->values();

        $content['data'] = $filteredUsers;
        set_response($response, $content);
        return $response;
    }
}
