<?php

namespace Inside\AGFR\Console;

use Illuminate\Console\Command;
use Illuminate\Support\Str;
use Inside\Host\Bridge\BridgeContent;
use League\Csv\Exception;
use League\Csv\Reader;

class ImportGedCommand extends Command
{
    /**
     * @var string
     */
    protected $name = 'agfr:ged:import';

    /**
     * @var string
     */
    protected $signature = 'agfr:ged:import {--d|delete-before}';

    /**
     * @var string
     */
    protected $description = 'Import catalogs';

    public function handle(): void
    {
        if ($this->option('delete-before')) {
            $this->getOutput()->writeln('<comment>Deleting current ged contents before import</comment>');

            $this->call('agfr:ged:purge');
        }

        $contentTypes = ['folders', 'articles'];

        foreach ($contentTypes as $contentType) {
            $this->runImport($contentType);
        }
    }

    protected function runImport(string $contentType, bool $rerun = false): void
    {
        if ($rerun) {
            $this->getOutput()->writeln(sprintf('<comment>Rerunning import for %s content type</comment>', $contentType));
        } else {
            $this->getOutput()->writeln(sprintf('<comment>Running import for %s content type</comment>', $contentType));
        }

        $bridge = new BridgeContent();

        $config = config('contents_import.' . $contentType);

        $otherFiles = [];
        if (array_key_exists('other_files', $config['settings'])) {
            foreach ($config['settings']['other_files'] as $key => $file) {
                $otherFiles[$key] = $this->getCsvRecords($file);
            }
        }

        $records = $this->getCsvRecords($config);

        $bar = $this->getOutput()->createProgressBar(count($records));
        $bar->setFormat("%message%\n %current%/%max% [%bar%] %percent:3s%%");

        foreach ($records as $record) {
            $data = [
                'type' => 'node',
                'langcode' => env('APP_LOCALE'),
                'bundle' => $contentType,
            ];

            foreach ($config['fields'] as $modelField => $fileField) {
                if ($fileField instanceof \Closure) {
                    $data[$modelField] = $fileField($record, $bridge, $otherFiles);
                } else {
                    $data[$modelField] = $record[$fileField];
                }
            }

            $bar->setMessage(sprintf('Importing %s content [%s] (%s)', $contentType, trim($data['title']), $data['old_id']));
            $bar->advance();

            if (empty($data['title'])) {
                continue;
            }

            if (array_key_exists('id', $config['settings']) && !empty($config['settings']['id']) && !empty($data[$config['settings']['id']])) {
                $query = call_user_func(type_to_class($contentType) . '::query');

                $content = $query->where($config['settings']['id'], $data[$config['settings']['id']])->first();

                if ($content) {
                    $data['uuid'] = $content->uuid;
                }
            }

            try {
                $bridge->contentInsert($contentType, $data, !array_key_exists('uuid', $data));
            } catch (\Exception $e) {
                dd($e->getMessage());
            }
        }

        if (!$rerun && $config['settings']['run_twice']) {
            $this->runImport($contentType, true);
        }
    }

    /**
     * We take the csv file and put it in an array with the right keys and values
     * @throws Exception
     */
    protected function getCsvRecords(mixed $config): array
    {
        $filePath = $config['settings']['path'];
        $csv = Reader::createFromPath($filePath, 'r');
        $encoding = mb_detect_encoding($csv->getContent(), mb_list_encodings(), true);

        if ($encoding !== 'UTF8') {
            $csv->setOutputBOM(Reader::BOM_UTF8);
            $csv->addStreamFilter('convert.iconv.' . $encoding . '/UTF-8');
        }

        $csv->setDelimiter($config['settings']['delimiter']);
        $csv->setHeaderOffset(0);

        $headers = array_map(
            function ($item) {
                return Str::lower(trim($item));
            },
            $csv->getHeader()
        );

        $records = iterator_to_array($csv->getRecords($headers));

        if (array_key_exists('columns', $config)) {
            $temp = [];

            foreach ($records as $record) {
                $key = $record[$config['settings']['id']];

                if (!array_key_exists($key, $temp)) {
                    $temp[$key] = [];
                }

                if ($config['settings']['multiple']) {
                    $temp[$key][] = array_only($record, $config['columns']);
                } else {
                    $temp[$key] = array_only($record, $config['columns']);
                }
            }

            return $temp;
        }

        return $records;
    }
}
