<?php

namespace Inside\AGFR\Console;

use Illuminate\Console\Command;
use Inside\Content\Models\Contents\FranchisedSpaces;
use Inside\Content\Models\Contents\Users;
use Inside\Host\Bridge\BridgeContent;
use League\Csv\Exception;
use League\Csv\Reader;
use League\Flysystem\FileNotFoundException;

class ImportFranchisedSpaceCommand extends Command
{
    /**
     * @var string
     */
    protected $name = 'agfr:franchised:import';

    /**
     * @var string
     */
    protected $signature = 'agfr:franchised:import { --P|path=}';

    /**
     * @var string
     */
    protected $description = 'Import franchised space';

    protected string $filePath = __DIR__ . '/../../../../../imports/franchised_space.csv';

    /**
     * @throws FileNotFoundException|Exception
     */
    public function handle(): void
    {
        if (!file_exists($this->filePath)) {
            throw new FileNotFoundException($this->filePath);
        }

        $records = $this->getCsvRecords();
        $this->writeln(sprintf('<info>Il y a [%d] catalogues %s à importer dans le csv %s</info>', count($records), 'franchised space', pathinfo($this->filePath)['basename']));

        $this->buildFranchisedSpace($records);
    }

    /**
     * @throws Exception
     */
    protected function getCsvRecords(): array
    {
        $csv = Reader::createFromPath($this->filePath, 'r');
        $encoding = mb_detect_encoding($csv->getContent(), mb_list_encodings(), true);

        if ($encoding !== 'UTF8') {
            $csv->setOutputBOM(Reader::BOM_UTF8);
            $csv->addStreamFilter('convert.iconv.' . $encoding . '/UTF-8');
        }

        $csv->setDelimiter(';');
        $csv->setHeaderOffset(0);

        $headers = array_map(
            function ($item) {
                return trim(strtolower($item));
            },
            $csv->getHeader()
        );

        return iterator_to_array($csv->getRecords($headers));
    }

    /**
     * @throws \Exception
     */
    protected function buildFranchisedSpace(array $records): void
    {
        $bridge = new BridgeContent();

        foreach ($records as $record) {
            foreach (config('franchised_space.franchised_spaces.fields') as $modelField => $fileField) {
                if ($fileField instanceof \Closure) {
                    $data[$modelField] = $fileField($record, $bridge);
                } else {
                    $data[$modelField] = $record[$fileField];
                }
            }

            if (empty($data['title'])) {
                continue;
            }

            $this->insertFranchisedSpace($data, $bridge);
        }
    }

    /**
     * @throws \Exception
     */
    protected function insertFranchisedSpace(array $data, BridgeContent $bridge): void
    {
        $query = call_user_func(type_to_class($data['bundle']) . '::query');

        if (!$query->where('title', $data['title'])->exists()) {
            $bridge->contentInsert($data['bundle'], $data);
            $bridge->contentUpdate('users', [
                'uuid' => $data['franchised'][0],
                'franchised_spaces' => FranchisedSpaces::where('title', $data['title'])->pluck('uuid')->first()
            ]);
        }
    }

    /**
     * Write and return to new line
     */
    protected function writeln(string $message): void
    {
        $this->getOutput()->writeln($message);
    }
}
