<?php

namespace Inside\AFP2\Services;

use Illuminate\Support\Facades\Log;
use Illuminate\Support\Facades\Storage;
use Inside\Content\Models\Contents\News;
use Inside\Content\Models\Contents\NewsCategories;
use Inside\Host\Bridge\BridgeContent;
use Inside\Host\Bridge\BridgeContentType;

class Afp2NewsCategoriesCleaner
{
    private const UUID_HOST = 0;
    private const ACTION = 6;
    private const NEW_UUID = 7;

    private BridgeContent $bridgeContent;

    private array $categories = [
        'to_delete' => [],
        'to_keep' => []
    ];

    public function __construct()
    {
    }

    public function execute(): void
    {
        $this->bridgeContent = new BridgeContent();
        $filePth = "clean/categories.csv";
        if (!Storage::disk('local')->exists($filePth)) {
            throw new \Exception("File not found: " . $filePth);
        }
        $file = Storage::disk('local')->path($filePth); // @phpstan-ignore-line
        $handle = fopen($file, 'r');
        if ($handle === false) {
            throw new \Exception("Failed to open file: " . $file);
        }
        $line = 1;
        $titles = [];
        while (($data = fgetcsv($handle, null, ';')) !== false) {
            if ($line === 1) {
                $line++;
                continue;
            }
            $newsCategories = NewsCategories::query()->where(
                'uuid_host',
                $data[self::UUID_HOST]
            )->get();

            if (str($data[self::ACTION])->lower()->toString() === 'a garder') {
                $this->categories['to_keep'][$data[self::UUID_HOST]] = $newsCategories->mapWithKeys(
                    fn (NewsCategories $category) => [$category->langcode => $category]
                );
            } elseif (str($data[self::ACTION])->lower()->toString() === 'a supprimer') {
                $this->categories['to_delete'][] = [
                    'from' => $data[self::UUID_HOST],
                    'to' => $data[self::NEW_UUID],
                    'deletable' => $newsCategories->mapWithKeys(
                        fn (NewsCategories $category) => [$category->langcode => $category]
                    )
                ];
            } else {
                continue;
            }
        }
        fclose($handle);

        $this->migrateCategories();
    }

    private function migrateCategories(): void
    {
        foreach ($this->categories['to_delete'] as $category) {
            try {
                // vérifier si la catégorie de destination existe
                $this->checkCategories($category);

                // Migrer les actu vers la nouvelle catégorie
                $this->moveNewsFromOldCategoriesToNewCategories($category);
            } catch (\Exception $e) {
                Log::error("Error migrating category from {$category['from']} to {$category['to']}: " . $e->getMessage());
            }
        }
    }

    private function checkCategories(array $category): void
    {
        $destination = $category['to'];
        $source = $category['from'];
        if (empty($this->categories['to_keep'][$destination])) {
            throw new \Exception("Destination category {$destination} does not exist.");
        }

        if ($category['deletable']->count() !== 2) {
            throw new \Exception("Source category {$source} does not exist or is not deletable.");
        }

        if ($this->categories['to_keep'][$destination]->count() !==2) {
            throw new \Exception("Destination category {$destination} does not have exactly two entries.");
        }
    }


    private function moveNewsFromOldCategoriesToNewCategories(array $category): void
    {
        $destination = $category['to'];
        $source = $category['from'];
        $newsCategories = $this->categories['to_keep'][$destination];
        $oldCategories = $category['deletable'];

        foreach ($oldCategories as $langcode => $oldCategory) {
            // Update news with the new category
            $newNewsCategory = $this->categories['to_keep'][$destination]->get($langcode);
            $news = News::query()
                ->where('langcode', $langcode)
                ->whereHas('newsCategories', fn ($builder) => $builder->where('uuid', $oldCategory->uuid))
                ->get();

            dump("{$news->count()} news for language {$langcode} from category {$oldCategory->title} [{$oldCategory->uuid}] to {$newNewsCategory->title} [{$newNewsCategory->uuid}].");
            foreach ($news as $counter => $newsItem) {
                dump("Proocessing item: ".($counter+1)." / {$news->count()}");
                $oldNewsCategories = $newsItem->newsCategories->pluck('uuid')->toArray();
                $newNewsCategories = collect($oldNewsCategories)->map(
                    fn ($item) => $item === $oldCategory->uuid ? $newNewsCategory->uuid : $item
                )->toArray();

                $this->bridgeContent->contentUpdate(
                    'news',
                    [
                        'uuid' => $newsItem->uuid,
                        'news_categories' => $newNewsCategories
                    ]
                );
            }
        }

        $categoryToDelete = $oldCategories->first()->uuid;

        $this->bridgeContent->contentDelete(
            'news_categories',
            $categoryToDelete
        );
    }
}
