<?php

declare(strict_types=1);

use Illuminate\Database\Migrations\Migration;
use Illuminate\Support\Facades\Artisan;
use Inside\Content\Models\Contents\Tools;
use Inside\Content\Models\Contents\Users;
use Inside\Host\Bridge\BridgeContent;
use Inside\Host\Migrations\CreateContentTypes;

final class SeedDefaulttools extends Migration
{
    use CreateContentTypes;

    public function up(): void
    {
        $this->bootUp(__FILE__);
        Artisan::call('models:rebuild');

        $tools = [
            ['fr' => 'Teams', 'en' => 'Teams'],
            ['fr' => 'Bibliothèque', 'en' => 'Library'],
            ['fr' => 'Chronos', 'en' => 'Chronos'],
            ['fr' => 'Iris 360', 'en' => 'Iris 360'],
            ['fr' => 'Notilus', 'en' => 'Notilus'],
        ];

        $bridgeContent = new BridgeContent();
        $tools = collect($tools)->map(fn (array $titles) => $this->getToolUuids($bridgeContent, $titles));

        $this->getBridge()?->contentTypeCreateOrUpdateField('users', 'tools', [
            'default' => $tools->pluck('uuid_host')->map(fn (string $uuid_host) => ['target_uuid' => $uuid_host])->all(),
        ]);

        Users::each(function (Users $user) use ($bridgeContent, $tools) {
            $defaultTools = $tools->pluck($user->langcode)
                ->merge($user->tools->pluck('uuid'))
                ->filter()
                ->unique()
                ->take($tools->count())
                ->values()
                ->all();

            try {
                $bridgeContent->contentUpdate('users', [
                    'uuid' => $user->uuid,
                    'tools' => $defaultTools,
                ]);
            } catch (Exception $exception) {
                $this->getConsole()->writeln("<error>Error for user $user->uuid ($user->email)</error>");
            }
        });

        $this->finishUp();
    }

    private function getToolUuids(BridgeContent $bridge, array $titles): array
    {
        $locale = config('app.locale');
        $languages = collect(list_languages())->reject($locale);

        $original = Tools::where(['title' => $titles[$locale], 'langcode' => $locale])->first();

        if (! $original instanceof Tools) {
            $original = $bridge->contentInsert('tools', ['title' => $titles[$locale], 'langcode' => $locale]);
            $original = Tools::findOrFail($original);
        }

        // collection of type [langcode => uuid]
        $uuids = $languages->mapWithKeys(function (string $langcode) use ($bridge, $titles, $original) {
            $data = [
                'langcode' => $langcode,
                'uuid_host' => $original->uuid_host,
            ];

            $uuid = Tools::where($data)->first()?->uuid ?? $bridge->contentInsert('tools', array_merge($data, ['title' => $titles[$langcode]]));
            return [$langcode => $uuid];
        });

        return $uuids->merge([$locale => $original->uuid, 'uuid_host' => $original->uuid_host])->all();
    }
}
