<?php

namespace Inside\AFAL\Http\Middleware;

use GuzzleHttp\Promise\FulfilledPromise;
use Illuminate\Http\Request;
use GuzzleHttp\Client as HttpClient;
use Illuminate\Http\Response;
use Illuminate\Support\Facades\Cache;
use Closure;

class WeatherDataMiddleware
{
    protected ?string $apiKey;

    public function __construct()
    {
        $this->apiKey = config('openweathermap.api_key');
    }

    public function handle(Request $request, Closure $next): mixed
    {
        $response = $next($request);

        if ($response instanceof Response && $request->method() === 'GET' && $request->path() === 'api/v1/content/weather_cities') {
            $data = $response->original;
            $this->addWeatherInfo($data['data']);
            $response->setContent($data);
        }

        return $response;
    }

    /**
     * Add 'temp' and 'icon' on the citiesx
     */
    protected function addWeatherInfo(array &$cities): void
    {
        // Fetch weather apis
        $client = new HttpClient();
        $promises = array_map(function ($city) use ($client) {
            if (empty($city['title'])) {
                return null;
            }

            // Build request url
            $url = 'https://api.openweathermap.org/data/2.5/weather?units=metric&q=';
            $url .= urlencode($city['title']);
            if (!empty($city['country'])) {
                $url .= ','.urlencode($city['country']);
            }
            $url .= '&APPID='.$this->apiKey;
            // Cache
            if ($cached = Cache::tags(['weather'])->get($url)) {
                return new FulfilledPromise($cached);
            }

            // Call api, decode body, save in cache
            return $client->sendAsync(new \GuzzleHttp\Psr7\Request('GET', $url))->then(function ($res) {
                return json_decode($res->getBody(), true);
            }, function () {
                return [];
            })->then(function ($res) use ($url) {
                Cache::tags(['weather'])->put($url, $res, 30);
                return $res;
            });
        }, $cities);

        // Wait till all the requests are finished
        $results = \GuzzleHttp\Promise\all($promises)->wait();
        // Fill cities data
        foreach ($cities as $i => &$city) {
            $data = $results[$i];

            // Temp
            $city['temp'] = $data['main']['temp'] ?? null;

            // Icon
            $city['icon'] = $data['weather'][0]['icon'] ?? null;

            // Timezone
            $city['timezone'] = $data['timezone'] ?? null;
        }
    }
}
